<?php

declare(strict_types=1);

namespace Drupal\trace_mail_log\Service;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Component\Uuid\UuidInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\Mime\Email;

/**
 * Service for logging mail operations.
 */
class MailLogService {

  /**
   * The database connection.
   */
  protected Connection $database;

  /**
   * The file system service.
   */
  protected FileSystemInterface $fileSystem;

  /**
   * The logger.
   */
  protected LoggerInterface $logger;

  /**
   * The config factory.
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The time service.
   */
  protected TimeInterface $time;

  /**
   * The UUID service.
   */
  protected UuidInterface $uuid;

  /**
   * Constructs a MailLogService.
   */
  public function __construct(
    Connection $database,
    FileSystemInterface $file_system,
    LoggerChannelFactoryInterface $logger_factory,
    ConfigFactoryInterface $config_factory,
    TimeInterface $time,
    UuidInterface $uuid,
  ) {
    $this->database = $database;
    $this->fileSystem = $file_system;
    $this->logger = $logger_factory->get('trace_mail_log');
    $this->configFactory = $config_factory;
    $this->time = $time;
    $this->uuid = $uuid;
  }

  /**
   * Checks if logging is enabled.
   */
  public function isEnabled(): bool {
    return (bool) $this->configFactory->get('trace_mail_log.settings')->get('enabled');
  }

  /**
   * Generates a new UUID for email tracking.
   */
  public function generateUuid(): string {
    return $this->uuid->generate();
  }

  /**
   * Logs a mail event.
   *
   * @param string $uuid
   *   The UUID correlating all events for this email.
   * @param string $event_type
   *   The event type: queued, sending, sent, failed, requeued.
   * @param string $status
   *   The status: pending, success, failed.
   * @param array $data
   *   Additional data including:
   *   - message_id: Email Message-ID header
   *   - mail_key: Drupal mail key
   *   - sender: From address
   *   - recipients: Array with 'to', 'cc', 'bcc' keys
   *   - subject: Email subject
   *   - transport_type: smtp, sendmail, native, null
   *   - response_code: SMTP code or exit code
   *   - response_message: Server response text
   *   - headers: Full email headers
   *   - body_html: HTML body
   *   - body_text: Text body
   *   - transcript: SMTP transcript
   */
  public function log(string $uuid, string $event_type, string $status, array $data = []): void {
    if (!$this->isEnabled()) {
      return;
    }

    $timestamp = $this->time->getRequestTime();
    $transcript_file = NULL;

    // Write full transcript to file if we have debug data.
    if (!empty($data['transcript']) || !empty($data['headers']) || !empty($data['body_html']) || !empty($data['body_text'])) {
      $transcript_file = $this->writeTranscriptFile($uuid, $event_type, $timestamp, $data);
    }

    // Insert database record.
    $fields = [
      'uuid' => $uuid,
      'message_id' => $data['message_id'] ?? NULL,
      'event_type' => $event_type,
      'status' => $status,
      'mail_key' => $data['mail_key'] ?? NULL,
      'sender' => $data['sender'] ?? NULL,
      'recipients' => isset($data['recipients']) ? json_encode($data['recipients']) : NULL,
      'subject' => isset($data['subject']) ? mb_substr($data['subject'], 0, 255) : NULL,
      'transport_type' => $data['transport_type'] ?? NULL,
      'response_code' => $data['response_code'] ?? NULL,
      'response_message' => isset($data['response_message']) ? mb_substr($data['response_message'], 0, 512) : NULL,
      'transcript_file' => $transcript_file,
      'created' => $timestamp,
    ];

    $this->database->insert('trace_mail_log')
      ->fields($fields)
      ->execute();

    // Log to watchdog.
    $this->logToWatchdog($event_type, $status, $data);
  }

  /**
   * Writes transcript data to a file.
   */
  protected function writeTranscriptFile(string $uuid, string $event_type, int $timestamp, array $data): ?string {
    $config = $this->configFactory->get('trace_mail_log.settings');
    $base_dir = $config->get('log_directory') ?? 'private://mail-logs';
    $log_body = (bool) $config->get('log_body');

    // Create directory structure: base/YYYY/MM/DD/
    $date_path = date('Y/m/d', $timestamp);
    $directory = $base_dir . '/' . $date_path;

    if (!$this->fileSystem->prepareDirectory($directory, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS)) {
      $this->logger->error('Failed to create mail log directory: @dir', ['@dir' => $directory]);
      return NULL;
    }

    // Build file content.
    $content = [];
    $content[] = '=' . str_repeat('=', 79);
    $content[] = 'MAIL LOG ENTRY';
    $content[] = '=' . str_repeat('=', 79);
    $content[] = '';
    $content[] = 'UUID: ' . $uuid;
    $content[] = 'Event: ' . $event_type;
    $content[] = 'Timestamp: ' . date('Y-m-d H:i:s', $timestamp);
    $content[] = 'Mail Key: ' . ($data['mail_key'] ?? 'N/A');
    $content[] = 'Transport: ' . ($data['transport_type'] ?? 'N/A');
    $content[] = '';

    // Recipients.
    if (!empty($data['recipients'])) {
      $content[] = '-' . str_repeat('-', 79);
      $content[] = 'RECIPIENTS';
      $content[] = '-' . str_repeat('-', 79);
      foreach (['to', 'cc', 'bcc'] as $type) {
        if (!empty($data['recipients'][$type])) {
          $content[] = strtoupper($type) . ': ' . implode(', ', (array) $data['recipients'][$type]);
        }
      }
      $content[] = '';
    }

    // Headers.
    if (!empty($data['headers'])) {
      $content[] = '-' . str_repeat('-', 79);
      $content[] = 'HEADERS';
      $content[] = '-' . str_repeat('-', 79);
      $content[] = $data['headers'];
      $content[] = '';
    }

    // Body (if enabled).
    if ($log_body) {
      if (!empty($data['body_html'])) {
        $content[] = '-' . str_repeat('-', 79);
        $content[] = 'HTML BODY';
        $content[] = '-' . str_repeat('-', 79);
        $content[] = $data['body_html'];
        $content[] = '';
      }
      if (!empty($data['body_text'])) {
        $content[] = '-' . str_repeat('-', 79);
        $content[] = 'TEXT BODY';
        $content[] = '-' . str_repeat('-', 79);
        $content[] = $data['body_text'];
        $content[] = '';
      }
    }

    // SMTP Transcript.
    if (!empty($data['transcript'])) {
      $content[] = '-' . str_repeat('-', 79);
      $content[] = 'SMTP TRANSCRIPT';
      $content[] = '-' . str_repeat('-', 79);
      $content[] = $data['transcript'];
      $content[] = '';
    }

    // Response.
    if (!empty($data['response_code']) || !empty($data['response_message'])) {
      $content[] = '-' . str_repeat('-', 79);
      $content[] = 'SERVER RESPONSE';
      $content[] = '-' . str_repeat('-', 79);
      $content[] = 'Code: ' . ($data['response_code'] ?? 'N/A');
      $content[] = 'Message: ' . ($data['response_message'] ?? 'N/A');
      $content[] = '';
    }

    // Error (if any).
    if (!empty($data['error'])) {
      $content[] = '-' . str_repeat('-', 79);
      $content[] = 'ERROR';
      $content[] = '-' . str_repeat('-', 79);
      $content[] = $data['error'];
      $content[] = '';
    }

    $content[] = '=' . str_repeat('=', 79);
    $content[] = 'END OF LOG ENTRY';
    $content[] = '=' . str_repeat('=', 79);

    // Write file.
    $filename = $uuid . '-' . $event_type . '.log';
    $filepath = $directory . '/' . $filename;

    try {
      $this->fileSystem->saveData(implode("\n", $content), $filepath, FileSystemInterface::EXISTS_REPLACE);
      return $filepath;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to write mail log file: @error', ['@error' => $e->getMessage()]);
      return NULL;
    }
  }

  /**
   * Logs event to Drupal watchdog.
   */
  protected function logToWatchdog(string $event_type, string $status, array $data): void {
    $recipients_str = '';
    if (!empty($data['recipients']['to'])) {
      $recipients_str = implode(', ', (array) $data['recipients']['to']);
    }

    $subject = $data['subject'] ?? 'No subject';
    $transport = $data['transport_type'] ?? 'unknown';
    $code = $data['response_code'] ?? '';

    switch ($event_type) {
      case 'queued':
        $this->logger->info('Email queued: "@subject" to @recipients', [
          '@subject' => $subject,
          '@recipients' => $recipients_str,
        ]);
        break;

      case 'sent':
        $this->logger->info('Email sent: "@subject" to @recipients (@transport @code)', [
          '@subject' => $subject,
          '@recipients' => $recipients_str,
          '@transport' => $transport,
          '@code' => $code,
        ]);
        break;

      case 'failed':
        $error = $data['error'] ?? $data['response_message'] ?? 'Unknown error';
        $this->logger->error('Email failed: "@subject" to @recipients - @error', [
          '@subject' => $subject,
          '@recipients' => $recipients_str,
          '@error' => $error,
        ]);
        break;

      case 'requeued':
        $attempt = $data['attempt'] ?? '?';
        $this->logger->warning('Email requeued (attempt @attempt): "@subject" to @recipients', [
          '@attempt' => $attempt,
          '@subject' => $subject,
          '@recipients' => $recipients_str,
        ]);
        break;
    }
  }

  /**
   * Extracts data from a Symfony Email message.
   */
  public function extractEmailData(Email $email, ?string $transport_dsn = NULL): array {
    $data = [
      'subject' => $email->getSubject(),
      'sender' => !empty($email->getFrom()) ? $email->getFrom()[0]->getAddress() : NULL,
      'recipients' => [
        'to' => array_map(fn($a) => $a->getAddress(), $email->getTo()),
        'cc' => array_map(fn($a) => $a->getAddress(), $email->getCc()),
        'bcc' => array_map(fn($a) => $a->getAddress(), $email->getBcc()),
      ],
      'body_html' => $email->getHtmlBody(),
      'body_text' => $email->getTextBody(),
      'headers' => $email->getHeaders()->toString(),
    ];

    // Determine transport type from DSN.
    if ($transport_dsn) {
      $data['transport_type'] = $this->parseTransportType($transport_dsn);
    }

    return $data;
  }

  /**
   * Parses transport type from DSN string.
   */
  protected function parseTransportType(string $dsn): string {
    if (str_starts_with($dsn, 'smtp://') || str_starts_with($dsn, 'smtps://')) {
      return 'smtp';
    }
    if (str_starts_with($dsn, 'sendmail://')) {
      return 'sendmail';
    }
    if (str_starts_with($dsn, 'native://')) {
      return 'native';
    }
    if (str_starts_with($dsn, 'null://')) {
      return 'null';
    }
    return 'unknown';
  }

  /**
   * Parses SMTP response code from transcript.
   */
  public function parseSmtpResponse(string $transcript): array {
    $result = [
      'code' => NULL,
      'message' => NULL,
    ];

    // Look for final response (usually after DATA command completes).
    // Pattern: "250 OK" or "250 2.0.0 Ok: queued as ABC123"
    if (preg_match('/^(\d{3})\s+(.+)$/m', $transcript, $matches)) {
      $result['code'] = (int) $matches[1];
      $result['message'] = trim($matches[2]);
    }

    // Get the last response code in the transcript.
    if (preg_match_all('/^(\d{3})[ -](.*)$/m', $transcript, $matches, PREG_SET_ORDER)) {
      $last = end($matches);
      $result['code'] = (int) $last[1];
      $result['message'] = trim($last[2]);
    }

    return $result;
  }

}
