<?php

declare(strict_types=1);

namespace Drupal\Tests\trace_mail_log\Kernel;

use Drupal\KernelTests\KernelTestBase;

/**
 * Tests schema installation and configuration.
 *
 * @group trace_mail_log
 */
class SchemaTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['trace_mail_log'];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installSchema('trace_mail_log', ['trace_mail_log']);
    $this->installConfig(['trace_mail_log']);
  }

  /**
   * Tests that the database table exists after installation.
   */
  public function testTableExists(): void {
    $database = \Drupal::database();
    $this->assertTrue($database->schema()->tableExists('trace_mail_log'));
  }

  /**
   * Tests that all expected columns exist in the table.
   */
  public function testTableColumns(): void {
    $database = \Drupal::database();
    $schema = $database->schema();

    $expected_columns = [
      'id',
      'uuid',
      'message_id',
      'event_type',
      'status',
      'mail_key',
      'sender',
      'recipients',
      'subject',
      'transport_type',
      'response_code',
      'response_message',
      'transcript_file',
      'created',
    ];

    foreach ($expected_columns as $column) {
      $this->assertTrue(
        $schema->fieldExists('trace_mail_log', $column),
        "Column '$column' should exist in trace_mail_log table"
      );
    }
  }

  /**
   * Tests that indexes exist on the table.
   */
  public function testTableIndexes(): void {
    $database = \Drupal::database();
    $schema = $database->schema();

    // Check primary key exists by trying to insert duplicate.
    // This indirectly tests the primary key.
    $database->insert('trace_mail_log')
      ->fields([
        'uuid' => 'test-uuid-1',
        'event_type' => 'sent',
        'status' => 'success',
        'created' => time(),
      ])
      ->execute();

    // Verify the auto-increment works.
    $id = $database->insert('trace_mail_log')
      ->fields([
        'uuid' => 'test-uuid-2',
        'event_type' => 'sent',
        'status' => 'success',
        'created' => time(),
      ])
      ->execute();

    $this->assertEquals(2, $id);

    // Check that indexes are used by running queries.
    // If indexes don't exist, these would be slow table scans.
    $database->select('trace_mail_log', 'm')
      ->fields('m', ['id'])
      ->condition('uuid', 'test-uuid-1')
      ->execute();

    $database->select('trace_mail_log', 'm')
      ->fields('m', ['id'])
      ->condition('status', 'success')
      ->execute();

    $database->select('trace_mail_log', 'm')
      ->fields('m', ['id'])
      ->condition('created', time() - 3600, '>')
      ->execute();

    // If we got here without errors, indexes are functional.
    $this->assertTrue(TRUE);
  }

  /**
   * Tests default configuration is installed.
   */
  public function testDefaultConfiguration(): void {
    $config = \Drupal::config('trace_mail_log.settings');

    $this->assertTrue($config->get('enabled'));
    $this->assertEquals(30, $config->get('retention_days'));
    $this->assertEquals('private://mail-logs', $config->get('log_directory'));
    $this->assertFalse($config->get('log_body'));
    $this->assertFalse($config->get('log_attachments'));
    $this->assertEquals('short', $config->get('date_format_list'));
    $this->assertEquals('medium', $config->get('date_format_detail'));
    $this->assertEquals(50, $config->get('items_per_page'));
  }

  /**
   * Tests configuration schema is valid.
   */
  public function testConfigurationSchema(): void {
    // This implicitly tests schema by checking that config typed data works.
    /** @var \Drupal\Core\Config\TypedConfigManagerInterface $typed_config */
    $typed_config = \Drupal::service('config.typed');

    $definition = $typed_config->getDefinition('trace_mail_log.settings');
    $this->assertNotNull($definition);
    $this->assertEquals('trace_mail_log.settings', $definition['label']);
  }

  /**
   * Tests configuration can be modified.
   */
  public function testConfigurationModification(): void {
    $config = \Drupal::configFactory()->getEditable('trace_mail_log.settings');

    $config->set('enabled', FALSE);
    $config->set('retention_days', 60);
    $config->set('date_format_list', 'medium');
    $config->set('items_per_page', 100);
    $config->save();

    // Reload config and verify changes.
    $config = \Drupal::config('trace_mail_log.settings');
    $this->assertFalse($config->get('enabled'));
    $this->assertEquals(60, $config->get('retention_days'));
    $this->assertEquals('medium', $config->get('date_format_list'));
    $this->assertEquals(100, $config->get('items_per_page'));
  }

  /**
   * Tests that data can be inserted into the table.
   */
  public function testDataInsertion(): void {
    $database = \Drupal::database();
    $now = time();

    $id = $database->insert('trace_mail_log')
      ->fields([
        'uuid' => 'test-uuid-insert',
        'message_id' => '<test@example.com>',
        'event_type' => 'sent',
        'status' => 'success',
        'mail_key' => 'test_module_test_key',
        'sender' => 'sender@example.com',
        'recipients' => json_encode(['to' => ['recipient@example.com']]),
        'subject' => 'Test Subject',
        'transport_type' => 'smtp',
        'response_code' => 250,
        'response_message' => 'OK',
        'transcript_file' => 'private://mail-logs/test.log',
        'created' => $now,
      ])
      ->execute();

    $this->assertGreaterThan(0, $id);

    // Verify data was stored correctly.
    $entry = $database->select('trace_mail_log', 'm')
      ->fields('m')
      ->condition('id', $id)
      ->execute()
      ->fetchObject();

    $this->assertEquals('test-uuid-insert', $entry->uuid);
    $this->assertEquals('<test@example.com>', $entry->message_id);
    $this->assertEquals('sent', $entry->event_type);
    $this->assertEquals('success', $entry->status);
    $this->assertEquals('Test Subject', $entry->subject);
    $this->assertEquals(250, $entry->response_code);
    $this->assertEquals($now, $entry->created);
  }

  /**
   * Tests that nullable fields accept NULL values.
   */
  public function testNullableFields(): void {
    $database = \Drupal::database();

    $id = $database->insert('trace_mail_log')
      ->fields([
        'uuid' => 'test-nullable',
        'event_type' => 'queued',
        'status' => 'pending',
        'message_id' => NULL,
        'mail_key' => NULL,
        'sender' => NULL,
        'recipients' => NULL,
        'subject' => NULL,
        'transport_type' => NULL,
        'response_code' => NULL,
        'response_message' => NULL,
        'transcript_file' => NULL,
        'created' => time(),
      ])
      ->execute();

    $entry = $database->select('trace_mail_log', 'm')
      ->fields('m')
      ->condition('id', $id)
      ->execute()
      ->fetchObject();

    $this->assertNull($entry->message_id);
    $this->assertNull($entry->mail_key);
    $this->assertNull($entry->sender);
    $this->assertNull($entry->recipients);
    $this->assertNull($entry->subject);
    $this->assertNull($entry->response_code);
  }

  /**
   * Tests service definitions are valid.
   */
  public function testServicesExist(): void {
    $this->assertTrue(\Drupal::hasService('trace_mail_log.mail_log_service'));
    $this->assertTrue(\Drupal::hasService('trace_mail_log.purge_service'));

    $mail_log_service = \Drupal::service('trace_mail_log.mail_log_service');
    $purge_service = \Drupal::service('trace_mail_log.purge_service');

    $this->assertInstanceOf(
      'Drupal\trace_mail_log\Service\MailLogService',
      $mail_log_service
    );
    $this->assertInstanceOf(
      'Drupal\trace_mail_log\Service\PurgeService',
      $purge_service
    );
  }

}
