<?php

declare(strict_types=1);

namespace Drupal\Tests\trace_mail_log\Unit;

use Drupal\Tests\UnitTestCase;
use Drupal\trace_mail_log\Service\MailLogService;

/**
 * Tests SMTP parsing methods in MailLogService.
 *
 * @coversDefaultClass \Drupal\trace_mail_log\Service\MailLogService
 * @group trace_mail_log
 */
class SmtpParserTest extends UnitTestCase {

  /**
   * The mail log service mock for testing protected methods.
   *
   * @var \Drupal\trace_mail_log\Service\MailLogService
   */
  protected MailLogService $service;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create a partial mock to test public methods that use parsing.
    // We need to mock all constructor dependencies.
    $database = $this->createMock('Drupal\Core\Database\Connection');
    $file_system = $this->createMock('Drupal\Core\File\FileSystemInterface');
    $logger_factory = $this->createMock('Drupal\Core\Logger\LoggerChannelFactoryInterface');
    $logger_factory->method('get')->willReturn($this->createMock('Psr\Log\LoggerInterface'));
    $config_factory = $this->createMock('Drupal\Core\Config\ConfigFactoryInterface');
    $time = $this->createMock('Drupal\Component\Datetime\TimeInterface');
    $uuid = $this->createMock('Drupal\Component\Uuid\UuidInterface');

    $this->service = new MailLogService(
      $database,
      $file_system,
      $logger_factory,
      $config_factory,
      $time,
      $uuid
    );
  }

  /**
   * Tests parseSmtpResponse with successful response.
   *
   * @covers ::parseSmtpResponse
   */
  public function testParseSmtpResponseSuccess(): void {
    $transcript = "220 smtp.example.com ESMTP\r\n250 OK\r\n250 2.0.0 Ok: queued as ABC123";

    $result = $this->service->parseSmtpResponse($transcript);

    $this->assertEquals(250, $result['code']);
    $this->assertEquals('2.0.0 Ok: queued as ABC123', $result['message']);
  }

  /**
   * Tests parseSmtpResponse with error response.
   *
   * @covers ::parseSmtpResponse
   */
  public function testParseSmtpResponseError(): void {
    $transcript = "220 smtp.example.com ESMTP\r\n550 Mailbox not found";

    $result = $this->service->parseSmtpResponse($transcript);

    $this->assertEquals(550, $result['code']);
    $this->assertEquals('Mailbox not found', $result['message']);
  }

  /**
   * Tests parseSmtpResponse with multi-line response.
   *
   * @covers ::parseSmtpResponse
   */
  public function testParseSmtpResponseMultiLine(): void {
    $transcript = <<<TRANSCRIPT
220 smtp.example.com ESMTP Postfix
250-smtp.example.com Hello client.example.com
250-SIZE 52428800
250-8BITMIME
250 DSN
354 End data with <CR><LF>.<CR><LF>
250 2.0.0 Ok: queued as 1234567890
TRANSCRIPT;

    $result = $this->service->parseSmtpResponse($transcript);

    $this->assertEquals(250, $result['code']);
    $this->assertStringContainsString('queued', $result['message']);
  }

  /**
   * Tests parseSmtpResponse with empty transcript.
   *
   * @covers ::parseSmtpResponse
   */
  public function testParseSmtpResponseEmpty(): void {
    $result = $this->service->parseSmtpResponse('');

    $this->assertNull($result['code']);
    $this->assertNull($result['message']);
  }

  /**
   * Tests parseSmtpResponse with temporary error.
   *
   * @covers ::parseSmtpResponse
   */
  public function testParseSmtpResponseTemporaryError(): void {
    $transcript = "220 smtp.example.com ESMTP\r\n421 Service temporarily unavailable";

    $result = $this->service->parseSmtpResponse($transcript);

    $this->assertEquals(421, $result['code']);
    $this->assertEquals('Service temporarily unavailable', $result['message']);
  }

  /**
   * Tests parseSmtpResponse with authentication failure.
   *
   * @covers ::parseSmtpResponse
   */
  public function testParseSmtpResponseAuthFailure(): void {
    $transcript = "220 smtp.example.com ESMTP\r\n535 5.7.8 Authentication credentials invalid";

    $result = $this->service->parseSmtpResponse($transcript);

    $this->assertEquals(535, $result['code']);
    $this->assertStringContainsString('Authentication', $result['message']);
  }

  /**
   * Data provider for transport type parsing tests.
   *
   * @return array
   *   Test cases with DSN and expected transport type.
   */
  public static function transportTypeProvider(): array {
    return [
      'smtp' => ['smtp://localhost:25', 'smtp'],
      'smtps' => ['smtps://smtp.gmail.com:465', 'smtp'],
      'smtp with auth' => ['smtp://user:pass@smtp.example.com:587', 'smtp'],
      'sendmail' => ['sendmail://default', 'sendmail'],
      'sendmail with path' => ['sendmail:///usr/sbin/sendmail', 'sendmail'],
      'native' => ['native://default', 'native'],
      'null' => ['null://null', 'null'],
      'unknown scheme' => ['custom://server', 'unknown'],
      'empty string' => ['', 'unknown'],
      'no scheme' => ['localhost:25', 'unknown'],
    ];
  }

  /**
   * Tests transport type parsing from DSN.
   *
   * @param string $dsn
   *   The DSN string.
   * @param string $expected
   *   The expected transport type.
   *
   * @covers ::parseTransportType
   * @dataProvider transportTypeProvider
   */
  public function testParseTransportType(string $dsn, string $expected): void {
    // Use reflection to access protected method.
    $reflection = new \ReflectionClass($this->service);
    $method = $reflection->getMethod('parseTransportType');
    $method->setAccessible(TRUE);

    $result = $method->invoke($this->service, $dsn);

    $this->assertEquals($expected, $result);
  }

}
