<?php

declare(strict_types=1);

namespace Drupal\Tests\trace_mail_log\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\Tests\trace_mail_log\Traits\MailLogTestTrait;
use Drupal\trace_mail_log\Controller\ExportController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\StreamedResponse;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;

/**
 * Tests the ExportController.
 *
 * @coversDefaultClass \Drupal\trace_mail_log\Controller\ExportController
 * @group trace_mail_log
 */
class ExportControllerTest extends KernelTestBase {

  use MailLogTestTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['trace_mail_log', 'symfony_mailer'];

  /**
   * The export controller.
   *
   * @var \Drupal\trace_mail_log\Controller\ExportController
   */
  protected ExportController $controller;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installSchema('trace_mail_log', ['trace_mail_log']);
    $this->installConfig(['trace_mail_log']);

    $this->controller = ExportController::create($this->container);
  }

  /**
   * Tests CSV export generates valid response.
   *
   * @covers ::exportAll
   * @covers ::generateCsv
   */
  public function testCsvExport(): void {
    $this->createMailLogEntry([
      'subject' => 'Test CSV Export',
      'recipients' => ['to' => ['test@example.com'], 'cc' => [], 'bcc' => []],
    ]);

    $response = $this->controller->exportAll('csv');

    $this->assertInstanceOf(StreamedResponse::class, $response);
    $this->assertEquals('text/csv; charset=utf-8', $response->headers->get('Content-Type'));
    $this->assertStringContainsString('attachment', $response->headers->get('Content-Disposition'));
    $this->assertStringContainsString('.csv', $response->headers->get('Content-Disposition'));
  }

  /**
   * Tests CSV export content.
   *
   * @covers ::exportAll
   * @covers ::generateCsv
   */
  public function testCsvExportContent(): void {
    $this->createMailLogEntry([
      'subject' => 'CSV Content Test',
      'sender' => 'from@example.com',
      'recipients' => [
        'to' => ['to@example.com'],
        'cc' => ['cc@example.com'],
        'bcc' => [],
      ],
      'response_code' => 250,
    ]);

    $response = $this->controller->exportAll('csv');

    // Capture streamed output.
    ob_start();
    $response->sendContent();
    $content = ob_get_clean();

    // Check header row (fputcsv quotes strings with spaces).
    $this->assertStringContainsString('ID,UUID,"Message ID"', $content);

    // Check data row.
    $this->assertStringContainsString('CSV Content Test', $content);
    $this->assertStringContainsString('from@example.com', $content);
    $this->assertStringContainsString('to@example.com', $content);
    $this->assertStringContainsString('cc@example.com', $content);
    $this->assertStringContainsString('250', $content);
  }

  /**
   * Tests JSON export generates valid response.
   *
   * @covers ::exportAll
   * @covers ::generateJson
   */
  public function testJsonExport(): void {
    $this->createMailLogEntry(['subject' => 'Test JSON Export']);

    $response = $this->controller->exportAll('json');

    $this->assertInstanceOf(Response::class, $response);
    $this->assertEquals('application/json; charset=utf-8', $response->headers->get('Content-Type'));
    $this->assertStringContainsString('.json', $response->headers->get('Content-Disposition'));
  }

  /**
   * Tests JSON export content structure.
   *
   * @covers ::exportAll
   * @covers ::generateJson
   */
  public function testJsonExportContent(): void {
    $this->createMailLogEntry([
      'subject' => 'JSON Content Test',
      'sender' => 'sender@example.com',
      'recipients' => [
        'to' => ['recipient@example.com'],
        'cc' => [],
        'bcc' => [],
      ],
      'event_type' => 'sent',
      'status' => 'success',
      'response_code' => 250,
    ]);

    $response = $this->controller->exportAll('json');
    $data = json_decode($response->getContent(), TRUE);

    $this->assertIsArray($data);
    $this->assertCount(1, $data);

    $entry = $data[0];
    $this->assertArrayHasKey('id', $entry);
    $this->assertArrayHasKey('uuid', $entry);
    $this->assertArrayHasKey('subject', $entry);
    $this->assertArrayHasKey('recipients', $entry);
    $this->assertArrayHasKey('created', $entry);
    $this->assertArrayHasKey('created_timestamp', $entry);

    $this->assertEquals('JSON Content Test', $entry['subject']);
    $this->assertEquals('sender@example.com', $entry['sender']);
    $this->assertEquals(['recipient@example.com'], $entry['recipients']['to']);
    $this->assertEquals('sent', $entry['event_type']);
    $this->assertEquals('success', $entry['status']);
    $this->assertEquals(250, $entry['response_code']);
  }

  /**
   * Tests filtered export with status filter.
   *
   * @covers ::exportFiltered
   * @covers ::buildQuery
   */
  public function testFilteredExportByStatus(): void {
    $this->createMailLogEntry(['status' => 'success']);
    $this->createMailLogEntry(['status' => 'success']);
    $this->createFailedMailLogEntry();

    $request = Request::create('/export/json', 'GET', ['status' => 'failed']);
    $response = $this->controller->exportFiltered('json', $request);

    $data = json_decode($response->getContent(), TRUE);
    $this->assertCount(1, $data);
    $this->assertEquals('failed', $data[0]['status']);
  }

  /**
   * Tests filtered export with event type filter.
   *
   * @covers ::exportFiltered
   * @covers ::buildQuery
   */
  public function testFilteredExportByEventType(): void {
    $this->createMailLogEntry(['event_type' => 'sent']);
    $this->createQueuedMailLogEntry();
    $this->createQueuedMailLogEntry();

    $request = Request::create('/export/json', 'GET', ['event_type' => 'queued']);
    $response = $this->controller->exportFiltered('json', $request);

    $data = json_decode($response->getContent(), TRUE);
    $this->assertCount(2, $data);
    foreach ($data as $entry) {
      $this->assertEquals('queued', $entry['event_type']);
    }
  }

  /**
   * Tests filtered export with recipient filter.
   *
   * @covers ::exportFiltered
   * @covers ::buildQuery
   */
  public function testFilteredExportByRecipient(): void {
    $this->createMailLogEntry([
      'recipients' => ['to' => ['alice@example.com'], 'cc' => [], 'bcc' => []],
    ]);
    $this->createMailLogEntry([
      'recipients' => ['to' => ['bob@example.com'], 'cc' => [], 'bcc' => []],
    ]);
    $this->createMailLogEntry([
      'recipients' => ['to' => ['alice@other.com'], 'cc' => [], 'bcc' => []],
    ]);

    $request = Request::create('/export/json', 'GET', ['recipient' => 'alice']);
    $response = $this->controller->exportFiltered('json', $request);

    $data = json_decode($response->getContent(), TRUE);
    $this->assertCount(2, $data);
  }

  /**
   * Tests export with multiple filters.
   *
   * @covers ::exportFiltered
   * @covers ::buildQuery
   */
  public function testFilteredExportMultipleFilters(): void {
    // Match: failed + contains "test".
    $this->createFailedMailLogEntry([
      'recipients' => ['to' => ['test@example.com'], 'cc' => [], 'bcc' => []],
    ]);
    // No match: success status.
    $this->createMailLogEntry([
      'recipients' => ['to' => ['test@example.com'], 'cc' => [], 'bcc' => []],
      'status' => 'success',
    ]);
    // No match: different recipient.
    $this->createFailedMailLogEntry([
      'recipients' => ['to' => ['other@example.com'], 'cc' => [], 'bcc' => []],
    ]);

    $request = Request::create('/export/json', 'GET', [
      'status' => 'failed',
      'recipient' => 'test',
    ]);
    $response = $this->controller->exportFiltered('json', $request);

    $data = json_decode($response->getContent(), TRUE);
    $this->assertCount(1, $data);
  }

  /**
   * Tests export with invalid format throws exception.
   *
   * @covers ::exportAll
   * @covers ::validateFormat
   */
  public function testInvalidFormatThrowsException(): void {
    $this->expectException(BadRequestHttpException::class);
    $this->expectExceptionMessage('Invalid export format');

    $this->controller->exportAll('xml');
  }

  /**
   * Tests export with empty results.
   *
   * @covers ::exportAll
   */
  public function testExportEmptyResults(): void {
    $response = $this->controller->exportAll('json');

    $data = json_decode($response->getContent(), TRUE);
    $this->assertIsArray($data);
    $this->assertEmpty($data);
  }

  /**
   * Tests export orders by created descending.
   *
   * @covers ::exportAll
   */
  public function testExportOrdersByCreatedDesc(): void {
    $this->createMailLogEntry([
      'subject' => 'Oldest',
      'created' => strtotime('-3 days'),
    ]);
    $this->createMailLogEntry([
      'subject' => 'Middle',
      'created' => strtotime('-2 days'),
    ]);
    $this->createMailLogEntry([
      'subject' => 'Newest',
      'created' => strtotime('-1 day'),
    ]);

    $response = $this->controller->exportAll('json');
    $data = json_decode($response->getContent(), TRUE);

    $this->assertEquals('Newest', $data[0]['subject']);
    $this->assertEquals('Middle', $data[1]['subject']);
    $this->assertEquals('Oldest', $data[2]['subject']);
  }

  /**
   * Tests CSV export handles special characters.
   *
   * @covers ::generateCsv
   */
  public function testCsvExportSpecialCharacters(): void {
    $this->createMailLogEntry([
      'subject' => 'Test "quotes" and, commas',
      'response_message' => "Line1\nLine2",
    ]);

    $response = $this->controller->exportAll('csv');

    ob_start();
    $response->sendContent();
    $content = ob_get_clean();

    // CSV should properly escape quotes and handle commas.
    $this->assertStringContainsString('"Test ""quotes"" and, commas"', $content);
  }

  /**
   * Tests JSON export handles null values.
   *
   * @covers ::generateJson
   */
  public function testJsonExportNullValues(): void {
    $this->createMailLogEntry([
      'response_code' => NULL,
      'response_message' => NULL,
      'message_id' => NULL,
    ]);

    $response = $this->controller->exportAll('json');
    $data = json_decode($response->getContent(), TRUE);

    $this->assertNull($data[0]['response_code']);
    $this->assertNull($data[0]['response_message']);
  }

}
