<?php

declare(strict_types=1);

namespace Drupal\trace_mail_log\Controller;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Pager\PagerManagerInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\ResponseHeaderBag;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Controller for mail log list and detail pages.
 */
class MailLogController extends ControllerBase {

  /**
   * Valid sort columns allowlist.
   */
  protected const SORT_COLUMNS = [
    'created' => 'created',
    'event_type' => 'event_type',
    'status' => 'status',
    'recipient' => 'recipients',
    'subject' => 'subject',
    'response_code' => 'response_code',
  ];

  /**
   * Constructs a MailLogController.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\File\FileSystemInterface $fileSystem
   *   The file system service.
   * @param \Drupal\Core\Pager\PagerManagerInterface $pagerManager
   *   The pager manager.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $dateFormatter
   *   The date formatter service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   */
  public function __construct(
    protected readonly Connection $database,
    protected readonly FileSystemInterface $fileSystem,
    protected readonly PagerManagerInterface $pagerManager,
    protected readonly DateFormatterInterface $dateFormatter,
    ConfigFactoryInterface $configFactory,
  ) {
    $this->configFactory = $configFactory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('database'),
      $container->get('file_system'),
      $container->get('pager.manager'),
      $container->get('date.formatter'),
      $container->get('config.factory'),
    );
  }

  /**
   * Displays the log list.
   */
  public function list(Request $request): array {
    $config = $this->configFactory->get('trace_mail_log.settings');

    // Get items per page from request or config.
    $allowed_limits = [25, 50, 100, 250];
    $default_limit = (int) ($config->get('items_per_page') ?: 50);
    $items_per_page = (int) $request->query->get('limit', $default_limit);
    if (!in_array($items_per_page, $allowed_limits, TRUE)) {
      $items_per_page = $default_limit;
    }

    // Get sort parameters.
    $sort = $request->query->get('sort', 'created');
    $order = $request->query->get('order', 'desc');

    // Validate sort column against allowlist.
    if (!isset(self::SORT_COLUMNS[$sort])) {
      $sort = 'created';
    }
    $db_column = self::SORT_COLUMNS[$sort];

    // Validate order direction.
    $order = strtolower($order) === 'asc' ? 'ASC' : 'DESC';

    // Build query.
    $query = $this->database->select('trace_mail_log', 'm')
      ->fields('m')
      ->orderBy($db_column, $order);

    // Apply filters.
    $filters = [
      'status' => $request->query->get('status'),
      'event_type' => $request->query->get('event_type'),
      'recipient' => $request->query->get('recipient'),
      'subject' => $request->query->get('subject'),
    ];

    if ($filters['status']) {
      $query->condition('status', $filters['status']);
    }
    if ($filters['event_type']) {
      $query->condition('event_type', $filters['event_type']);
    }
    if ($filters['recipient']) {
      $query->condition('recipients', '%' . $this->database->escapeLike($filters['recipient']) . '%', 'LIKE');
    }
    if ($filters['subject']) {
      $query->condition('subject', '%' . $this->database->escapeLike($filters['subject']) . '%', 'LIKE');
    }

    // Count total.
    $count_query = $query->countQuery();
    $total = (int) $count_query->execute()->fetchField();

    // Apply pager.
    $pager = $this->pagerManager->createPager($total, $items_per_page);
    $query->range($pager->getCurrentPage() * $items_per_page, $items_per_page);

    $results = $query->execute()->fetchAll();

    // Check if user can delete entries.
    $can_delete = $this->currentUser()->hasPermission('administer mail_log');

    // Get configured date format.
    $date_format = $config->get('date_format_list') ?: 'short';

    // Build rows.
    $rows = [];
    foreach ($results as $row) {
      $recipients = json_decode($row->recipients, TRUE);
      $to = implode(', ', $recipients['to'] ?? []);

      // Build operations links for dropbutton.
      $operations = [
        'view' => [
          'title' => $this->t('View'),
          'url' => Url::fromRoute('trace_mail_log.detail', ['id' => $row->id]),
        ],
      ];
      if ($can_delete) {
        $operations['delete'] = [
          'title' => $this->t('Delete'),
          'url' => Url::fromRoute('trace_mail_log.delete_entry', ['id' => $row->id]),
        ];
      }

      $rows[] = [
        'id' => $row->id,
        'date' => $this->dateFormatter->format((int) $row->created, $date_format),
        'event_type' => $row->event_type,
        'status' => $row->status,
        'recipient' => mb_strlen($to) > 50 ? mb_substr($to, 0, 50) . '...' : $to,
        'subject' => mb_strlen($row->subject ?? '') > 40 ? mb_substr($row->subject, 0, 40) . '...' : ($row->subject ?? ''),
        'response_code' => $row->response_code,
        'operations' => [
          '#type' => 'operations',
          '#links' => $operations,
        ],
      ];
    }

    // Build sort links for each column.
    $sort_links = $this->buildSortLinks($sort, $order, $filters, $items_per_page);

    // Build limit links to preserve current filters and sort.
    $limit_links = $this->buildLimitLinks($sort, $order, $filters, $allowed_limits);

    // Build export URLs with current filters.
    $export_query = array_filter($filters);
    $export_urls = [
      'csv_filtered' => Url::fromRoute('trace_mail_log.export', ['format' => 'csv'], ['query' => $export_query])->toString(),
      'json_filtered' => Url::fromRoute('trace_mail_log.export', ['format' => 'json'], ['query' => $export_query])->toString(),
      'csv_all' => Url::fromRoute('trace_mail_log.export_all', ['format' => 'csv'])->toString(),
      'json_all' => Url::fromRoute('trace_mail_log.export_all', ['format' => 'json'])->toString(),
    ];

    return [
      '#theme' => 'mail_log_list',
      '#rows' => $rows,
      '#filters' => $filters,
      '#pager' => ['#type' => 'pager'],
      '#can_delete' => $can_delete,
      '#sort' => $sort,
      '#order' => strtolower($order),
      '#sort_links' => $sort_links,
      '#items_per_page' => $items_per_page,
      '#allowed_limits' => $allowed_limits,
      '#limit_links' => $limit_links,
      '#export_urls' => $export_urls,
    ];
  }

  /**
   * Builds sort links for column headers.
   *
   * @param string $current_sort
   *   The current sort column.
   * @param string $current_order
   *   The current sort order (ASC or DESC).
   * @param array $filters
   *   Current filter values.
   * @param int $limit
   *   Current items per page.
   *
   * @return array
   *   Array of sort link URLs keyed by column name.
   */
  protected function buildSortLinks(string $current_sort, string $current_order, array $filters, int $limit): array {
    $links = [];
    $base_query = array_filter($filters);
    $base_query['limit'] = $limit;

    foreach (array_keys(self::SORT_COLUMNS) as $column) {
      $query = $base_query;
      $query['sort'] = $column;

      // Toggle order if clicking the same column.
      if ($column === $current_sort) {
        $query['order'] = ($current_order === 'ASC') ? 'desc' : 'asc';
      }
      else {
        $query['order'] = 'asc';
      }

      $links[$column] = Url::fromRoute('trace_mail_log.list', [], ['query' => $query])->toString();
    }

    return $links;
  }

  /**
   * Builds limit links for items per page selector.
   *
   * @param string $current_sort
   *   The current sort column.
   * @param string $current_order
   *   The current sort order (ASC or DESC).
   * @param array $filters
   *   Current filter values.
   * @param array $allowed_limits
   *   Array of allowed limit values.
   *
   * @return array
   *   Array of limit link URLs keyed by limit value.
   */
  protected function buildLimitLinks(string $current_sort, string $current_order, array $filters, array $allowed_limits): array {
    $links = [];
    $base_query = array_filter($filters);
    $base_query['sort'] = $current_sort;
    $base_query['order'] = strtolower($current_order);

    foreach ($allowed_limits as $limit) {
      $query = $base_query;
      $query['limit'] = $limit;
      $links[$limit] = Url::fromRoute('trace_mail_log.list', [], ['query' => $query])->toString();
    }

    return $links;
  }

  /**
   * Displays a log entry detail.
   */
  public function detail(int $id): array {
    $config = $this->configFactory->get('trace_mail_log.settings');
    $date_format = $config->get('date_format_detail') ?: 'medium';

    $entry = $this->database->select('trace_mail_log', 'm')
      ->fields('m')
      ->condition('id', $id)
      ->execute()
      ->fetchObject();

    if (!$entry) {
      throw new NotFoundHttpException();
    }

    // Get all events for this email (same UUID), including current entry.
    $timeline = [];
    if ($entry->uuid) {
      $timeline_query = $this->database->select('trace_mail_log', 'm')
        ->fields('m', ['id', 'event_type', 'status', 'created'])
        ->condition('uuid', $entry->uuid)
        ->orderBy('created', 'ASC');

      $timeline_results = $timeline_query->execute()->fetchAll();
      foreach ($timeline_results as $item) {
        $timeline[] = [
          'id' => (int) $item->id,
          'event_type' => $item->event_type,
          'status' => $item->status,
          'created' => $this->dateFormatter->format((int) $item->created, $date_format),
          'is_current' => ((int) $item->id === $id),
        ];
      }
    }

    // Load transcript file content if available.
    $transcript_content = NULL;
    if ($entry->transcript_file) {
      $realpath = $this->fileSystem->realpath($entry->transcript_file);
      if ($realpath && file_exists($realpath)) {
        $transcript_content = file_get_contents($realpath);
      }
    }

    $recipients = json_decode($entry->recipients, TRUE) ?? [];

    // Check if user can delete entries.
    $can_delete = $this->currentUser()->hasPermission('administer mail_log');

    return [
      '#theme' => 'mail_log_detail',
      '#entry' => [
        'id' => $entry->id,
        'uuid' => $entry->uuid,
        'message_id' => $entry->message_id,
        'event_type' => $entry->event_type,
        'status' => $entry->status,
        'mail_key' => $entry->mail_key,
        'sender' => $entry->sender,
        'recipients' => $recipients,
        'subject' => $entry->subject,
        'transport_type' => $entry->transport_type,
        'response_code' => $entry->response_code,
        'response_message' => $entry->response_message,
        'created' => $this->dateFormatter->format((int) $entry->created, $date_format),
        'transcript_file' => $entry->transcript_file,
      ],
      '#transcript_content' => $transcript_content,
      '#timeline' => $timeline,
      '#download_url' => $entry->transcript_file ? Url::fromRoute('trace_mail_log.download', ['id' => $id])->toString() : NULL,
      '#can_delete' => $can_delete,
    ];
  }

  /**
   * Downloads transcript file.
   */
  public function download(int $id): BinaryFileResponse {
    $entry = $this->database->select('trace_mail_log', 'm')
      ->fields('m', ['transcript_file', 'uuid', 'event_type'])
      ->condition('id', $id)
      ->execute()
      ->fetchObject();

    if (!$entry || !$entry->transcript_file) {
      throw new NotFoundHttpException();
    }

    $realpath = $this->fileSystem->realpath($entry->transcript_file);
    if (!$realpath || !file_exists($realpath)) {
      throw new NotFoundHttpException();
    }

    $response = new BinaryFileResponse($realpath);
    $filename = $entry->uuid . '-' . $entry->event_type . '.log';
    $response->setContentDisposition(ResponseHeaderBag::DISPOSITION_ATTACHMENT, $filename);

    return $response;
  }

}
