<?php

declare(strict_types=1);

namespace Drupal\trace_mail_log\Form;

use Drupal\Core\Database\Connection;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\trace_mail_log\Service\PurgeService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Confirmation form for deleting a single mail log entry.
 */
class DeleteEntryForm extends ConfirmFormBase {

  /**
   * The log entry ID to delete.
   */
  protected int $id;

  /**
   * Constructs a DeleteEntryForm.
   *
   * @param \Drupal\trace_mail_log\Service\PurgeService $purgeService
   *   The purge service.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $dateFormatter
   *   The date formatter service.
   */
  public function __construct(
    protected readonly PurgeService $purgeService,
    protected readonly Connection $database,
    protected readonly DateFormatterInterface $dateFormatter,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('trace_mail_log.purge_service'),
      $container->get('database'),
      $container->get('date.formatter'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'trace_mail_log_delete_entry_form';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Are you sure you want to delete this mail log entry?');
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription(): TranslatableMarkup {
    // Get entry details for display.
    $entry = $this->database->select('trace_mail_log', 'm')
      ->fields('m', ['subject', 'recipients', 'created'])
      ->condition('id', $this->id)
      ->execute()
      ->fetchObject();

    if ($entry) {
      $recipients = json_decode($entry->recipients, TRUE);
      $to = implode(', ', $recipients['to'] ?? []);
      $date = $this->dateFormatter->format((int) $entry->created, 'custom', 'Y-m-d H:i:s');
      $subject = $entry->subject ?? $this->t('N/A');
      $recipient = $to ?: $this->t('N/A');

      return $this->t('This will permanently delete the log entry:<br><strong>Subject:</strong> @subject<br><strong>To:</strong> @recipient<br><strong>Date:</strong> @date<br>This action cannot be undone.', [
        '@subject' => $subject,
        '@recipient' => $recipient,
        '@date' => $date,
      ]);
    }

    return $this->t('This will permanently delete the log entry and its associated transcript file. This action cannot be undone.');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl(): Url {
    return new Url('trace_mail_log.list');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText() {
    return $this->t('Delete');
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?int $id = NULL): array {
    $this->id = $id;

    // Verify the entry exists.
    $exists = $this->database->select('trace_mail_log', 'm')
      ->condition('id', $id)
      ->countQuery()
      ->execute()
      ->fetchField();

    if (!$exists) {
      $this->messenger()->addError($this->t('The mail log entry does not exist.'));
      return $this->redirect('trace_mail_log.list')->send();
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    if ($this->purgeService->deleteSingle($this->id)) {
      $this->messenger()->addStatus($this->t('The mail log entry has been deleted.'));
    }
    else {
      $this->messenger()->addError($this->t('Failed to delete the mail log entry.'));
    }

    $form_state->setRedirectUrl($this->getCancelUrl());
  }

}
