<?php

declare(strict_types=1);

namespace Drupal\trace_mail_log;

use Drupal\Component\Plugin\Discovery\CachedDiscoveryInterface;
use Drupal\Core\Cache\CacheableDependencyInterface;
use Drupal\Core\Mail\MailManagerInterface;
use Drupal\trace_mail_log\Service\MailLogService;

/**
 * Decorates the mail manager to log all outgoing emails.
 *
 * This decorator transparently intercepts all mail sent through Drupal's
 * mail system, logging before and after sending without requiring any
 * configuration changes from the user.
 */
class MailManagerDecorator implements MailManagerInterface, CachedDiscoveryInterface, CacheableDependencyInterface {

  /**
   * Constructs a MailManagerDecorator.
   *
   * @param \Drupal\Core\Mail\MailManagerInterface $innerMailManager
   *   The decorated mail manager.
   * @param \Drupal\trace_mail_log\Service\MailLogService $mailLogService
   *   The mail log service.
   */
  public function __construct(
    protected readonly MailManagerInterface $innerMailManager,
    protected readonly MailLogService $mailLogService,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function mail($module, $key, $to, $langcode, $params = [], $reply = NULL, $send = TRUE): array {
    // If not sending or logging disabled, just pass through.
    if (!$send || !$this->mailLogService->isEnabled()) {
      return $this->innerMailManager->mail($module, $key, $to, $langcode, $params, $reply, $send);
    }

    // Generate UUID for tracking.
    $uuid = $this->mailLogService->generateUuid();

    // Call the inner mail manager to build and send.
    $result = $this->innerMailManager->mail($module, $key, $to, $langcode, $params, $reply, $send);

    // Extract data from the result message array.
    $data = $this->extractMailData($result, $module, $key);

    if (!empty($result['result'])) {
      // Log "sent" event.
      $data['response_message'] = 'Mail accepted for delivery';
      $this->mailLogService->log($uuid, 'sent', 'success', $data);
    }
    else {
      // Log "failed" event.
      $data['error'] = $result['send'] ? 'Mail sending failed' : 'Mail sending was cancelled';
      $this->mailLogService->log($uuid, 'failed', 'failed', $data);
    }

    return $result;
  }

  /**
   * Extracts mail data from Drupal message array.
   *
   * @param array $message
   *   The Drupal mail message array.
   * @param string $module
   *   The module sending the mail.
   * @param string $key
   *   The mail key.
   *
   * @return array
   *   Extracted data for logging.
   */
  protected function extractMailData(array $message, string $module, string $key): array {
    $data = [
      'mail_key' => $message['id'] ?? "{$module}_{$key}",
      'module' => $message['module'] ?? $module,
      'subject' => $message['subject'] ?? NULL,
      'sender' => $message['from'] ?? ($message['headers']['From'] ?? NULL),
      'transport_type' => 'drupal_mail',
    ];

    // Extract recipients.
    // Drupal core's mail system uses comma-separated strings for addresses.
    $recipients = [];
    if (!empty($message['to'])) {
      $recipients['to'] = array_map('trim', explode(',', $message['to']));
    }
    if (!empty($message['headers']['Cc'])) {
      $cc = $message['headers']['Cc'];
      $recipients['cc'] = array_map('trim', explode(',', $cc));
    }
    if (!empty($message['headers']['Bcc'])) {
      $bcc = $message['headers']['Bcc'];
      $recipients['bcc'] = array_map('trim', explode(',', $bcc));
    }
    $data['recipients'] = $recipients;

    // Include headers if available.
    if (!empty($message['headers'])) {
      $headers = [];
      foreach ($message['headers'] as $name => $value) {
        $headers[] = "{$name}: {$value}";
      }
      $data['headers'] = implode("\n", $headers);
    }

    // Include body if configured to log body content.
    if (!empty($message['body']) && $this->mailLogService->shouldLogBody()) {
      $body = is_array($message['body']) ? implode("\n\n", $message['body']) : $message['body'];
      $data['body_text'] = $body;
    }

    return $data;
  }

  /**
   * {@inheritdoc}
   */
  public function getInstance(array $options) {
    return $this->innerMailManager->getInstance($options);
  }

  /**
   * {@inheritdoc}
   */
  public function createInstance($plugin_id, array $configuration = []) {
    return $this->innerMailManager->createInstance($plugin_id, $configuration);
  }

  /**
   * {@inheritdoc}
   */
  public function getDefinition($plugin_id, $exception_on_invalid = TRUE) {
    return $this->innerMailManager->getDefinition($plugin_id, $exception_on_invalid);
  }

  /**
   * {@inheritdoc}
   */
  public function getDefinitions() {
    return $this->innerMailManager->getDefinitions();
  }

  /**
   * {@inheritdoc}
   */
  public function hasDefinition($plugin_id) {
    return $this->innerMailManager->hasDefinition($plugin_id);
  }

  /**
   * {@inheritdoc}
   */
  public function clearCachedDefinitions(): void {
    if ($this->innerMailManager instanceof CachedDiscoveryInterface) {
      $this->innerMailManager->clearCachedDefinitions();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function useCaches($use_caches = FALSE): void {
    if ($this->innerMailManager instanceof CachedDiscoveryInterface) {
      $this->innerMailManager->useCaches($use_caches);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheContexts(): array {
    if ($this->innerMailManager instanceof CacheableDependencyInterface) {
      return $this->innerMailManager->getCacheContexts();
    }
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheTags(): array {
    if ($this->innerMailManager instanceof CacheableDependencyInterface) {
      return $this->innerMailManager->getCacheTags();
    }
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheMaxAge(): int {
    if ($this->innerMailManager instanceof CacheableDependencyInterface) {
      return $this->innerMailManager->getCacheMaxAge();
    }
    return 0;
  }

}
