<?php

declare(strict_types=1);

namespace Drupal\trace_mail_log\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Psr\Log\LoggerInterface;

/**
 * Service for purging old mail log entries.
 */
class PurgeService {

  /**
   * The logger.
   */
  protected readonly LoggerInterface $logger;

  /**
   * Constructs a PurgeService.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\File\FileSystemInterface $fileSystem
   *   The file system service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   The logger factory.
   */
  public function __construct(
    protected readonly Connection $database,
    protected readonly FileSystemInterface $fileSystem,
    protected readonly ConfigFactoryInterface $configFactory,
    LoggerChannelFactoryInterface $loggerFactory,
  ) {
    $this->logger = $loggerFactory->get('trace_mail_log');
  }

  /**
   * Purges logs older than the retention period.
   *
   * @return int
   *   Number of entries purged.
   */
  public function purge(): int {
    $config = $this->configFactory->get('trace_mail_log.settings');
    $retention_days = (int) $config->get('retention_days');

    if ($retention_days <= 0) {
      return 0;
    }

    $cutoff = strtotime("-{$retention_days} days");
    $count = 0;

    // Get entries to delete (need file paths before deleting).
    $query = $this->database->select('trace_mail_log', 'm')
      ->fields('m', ['id', 'transcript_file'])
      ->condition('created', $cutoff, '<');

    $results = $query->execute()->fetchAll();

    foreach ($results as $row) {
      // Delete transcript file if it exists.
      if (!empty($row->transcript_file)) {
        $this->deleteTranscriptFile($row->transcript_file);
      }
      $count++;
    }

    // Delete database records.
    if ($count > 0) {
      $this->database->delete('trace_mail_log')
        ->condition('created', $cutoff, '<')
        ->execute();

      $this->logger->info('Purged @count mail log entries older than @days days.', [
        '@count' => $count,
        '@days' => $retention_days,
      ]);

      // Clean up empty directories.
      $this->cleanupEmptyDirectories();
    }

    return $count;
  }

  /**
   * Deletes a transcript file.
   */
  protected function deleteTranscriptFile(string $filepath): void {
    try {
      $realpath = $this->fileSystem->realpath($filepath);
      if ($realpath && file_exists($realpath)) {
        $this->fileSystem->delete($filepath);
      }
    }
    catch (\Exception $e) {
      $this->logger->warning('Failed to delete transcript file @file: @error', [
        '@file' => $filepath,
        '@error' => $e->getMessage(),
      ]);
    }
  }

  /**
   * Cleans up empty date directories.
   */
  protected function cleanupEmptyDirectories(): void {
    $config = $this->configFactory->get('trace_mail_log.settings');
    $base_dir = $config->get('log_directory') ?? 'private://mail-logs';
    $realpath = $this->fileSystem->realpath($base_dir);

    if (!$realpath || !is_dir($realpath)) {
      return;
    }

    // Scan for empty day/month/year directories and remove them.
    $this->removeEmptyDirsRecursive($realpath);
  }

  /**
   * Recursively removes empty directories.
   */
  protected function removeEmptyDirsRecursive(string $dir): bool {
    if (!is_dir($dir)) {
      return FALSE;
    }

    $files = array_diff(scandir($dir), ['.', '..']);
    $empty = TRUE;

    foreach ($files as $file) {
      $path = $dir . '/' . $file;
      if (is_dir($path)) {
        if (!$this->removeEmptyDirsRecursive($path)) {
          $empty = FALSE;
        }
      }
      else {
        $empty = FALSE;
      }
    }

    // Don't remove the base directory.
    if ($empty && $dir !== $this->fileSystem->realpath($this->configFactory->get('trace_mail_log.settings')->get('log_directory'))) {
      rmdir($dir);
      return TRUE;
    }

    return $empty;
  }

  /**
   * Deletes a single log entry by ID.
   *
   * @param int $id
   *   The log entry ID.
   *
   * @return bool
   *   TRUE if entry was deleted, FALSE if not found.
   */
  public function deleteSingle(int $id): bool {
    // Get the entry to find transcript file.
    $entry = $this->database->select('trace_mail_log', 'm')
      ->fields('m', ['id', 'transcript_file'])
      ->condition('id', $id)
      ->execute()
      ->fetchObject();

    if (!$entry) {
      return FALSE;
    }

    // Delete transcript file if it exists.
    if (!empty($entry->transcript_file)) {
      $this->deleteTranscriptFile($entry->transcript_file);
    }

    // Delete database record.
    $this->database->delete('trace_mail_log')
      ->condition('id', $id)
      ->execute();

    $this->logger->info('Deleted mail log entry @id.', ['@id' => $id]);

    // Clean up empty directories.
    $this->cleanupEmptyDirectories();

    return TRUE;
  }

  /**
   * Deletes all log entries.
   *
   * @return int
   *   Number of entries deleted.
   */
  public function deleteAll(): int {
    // Get all entries with transcript files.
    $query = $this->database->select('trace_mail_log', 'm')
      ->fields('m', ['id', 'transcript_file']);

    $results = $query->execute()->fetchAll();
    $count = count($results);

    // Delete transcript files.
    foreach ($results as $row) {
      if (!empty($row->transcript_file)) {
        $this->deleteTranscriptFile($row->transcript_file);
      }
    }

    // Delete all database records.
    if ($count > 0) {
      $this->database->truncate('trace_mail_log')->execute();

      $this->logger->warning('Deleted all @count mail log entries.', ['@count' => $count]);

      // Clean up empty directories.
      $this->cleanupEmptyDirectories();
    }

    return $count;
  }

  /**
   * Gets statistics about the log.
   */
  public function getStats(): array {
    $stats = [
      'total' => 0,
      'today' => 0,
      'this_week' => 0,
      'failures_this_week' => 0,
    ];

    $today_start = strtotime('today');
    $week_start = strtotime('-7 days');

    // Total count.
    $stats['total'] = (int) $this->database->select('trace_mail_log', 'm')
      ->countQuery()
      ->execute()
      ->fetchField();

    // Today's sent.
    $stats['today'] = (int) $this->database->select('trace_mail_log', 'm')
      ->condition('event_type', 'sent')
      ->condition('created', $today_start, '>=')
      ->countQuery()
      ->execute()
      ->fetchField();

    // This week's sent.
    $stats['this_week'] = (int) $this->database->select('trace_mail_log', 'm')
      ->condition('event_type', 'sent')
      ->condition('created', $week_start, '>=')
      ->countQuery()
      ->execute()
      ->fetchField();

    // Failures this week.
    $stats['failures_this_week'] = (int) $this->database->select('trace_mail_log', 'm')
      ->condition('event_type', 'failed')
      ->condition('created', $week_start, '>=')
      ->countQuery()
      ->execute()
      ->fetchField();

    return $stats;
  }

}
