<?php

declare(strict_types=1);

namespace Drupal\trace_mail_log\Service;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Database\Connection;
use Drupal\webform\WebformSubmissionInterface;

/**
 * Service to check email sending status for webform submissions.
 */
class WebformEmailStatusService {

  /**
   * Status constants.
   */
  public const STATUS_SUCCESS = 'success';
  public const STATUS_FAILED = 'failed';
  public const STATUS_PENDING = 'pending';
  public const STATUS_UNKNOWN = 'unknown';
  public const STATUS_NO_HANDLER = 'no_handler';

  /**
   * Time window in seconds to search for email logs after submission.
   *
   * Due to queue processing delays, emails may be sent some time after
   * submission.
   */
  protected const TIME_WINDOW_SECONDS = 600;

  /**
   * Cached result of table existence check.
   */
  protected ?bool $tableExists = NULL;

  /**
   * Constructs a WebformEmailStatusService.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Cache\CacheBackendInterface $memoryCache
   *   The memory cache service.
   */
  public function __construct(
    protected readonly Connection $database,
    protected readonly CacheBackendInterface $memoryCache,
  ) {}

  /**
   * Gets the email status for a webform submission.
   *
   * @param \Drupal\webform\WebformSubmissionInterface $submission
   *   The webform submission.
   *
   * @return array{status: string, details: array}
   *   An array with 'status' and 'details' keys.
   */
  public function getEmailStatus(WebformSubmissionInterface $submission): array {
    $sid = (int) $submission->id();
    $cache_key = 'email_status:' . $sid;

    // Check memory cache.
    $cached = $this->memoryCache->get($cache_key);
    if ($cached) {
      return $cached->data;
    }

    $webform = $submission->getWebform();
    $handlers = $webform->getHandlers();

    // Check if there are any enabled email handlers.
    $email_handlers = [];
    foreach ($handlers as $handler_id => $handler) {
      if ($handler->getPluginId() === 'email' && $handler->isEnabled()) {
        $email_handlers[$handler_id] = $handler;
      }
    }

    if (empty($email_handlers)) {
      $result = [
        'status' => self::STATUS_NO_HANDLER,
        'details' => [],
      ];
      $this->memoryCache->set($cache_key, $result);
      return $result;
    }

    // Build mail keys for all email handlers.
    $mail_keys = [];
    foreach ($email_handlers as $handler_id => $handler) {
      $mail_keys[$handler_id] = 'webform_' . $webform->id() . '_' . $handler_id;
    }

    // Query trace_mail_log for emails matching these mail keys and time window.
    $submission_created = (int) $submission->getCreatedTime();
    $time_end = $submission_created + self::TIME_WINDOW_SECONDS;

    // Batch query for all mail keys at once (fixes N+1 problem).
    $statuses = $this->queryEmailStatusBatch($mail_keys, $submission_created, $time_end);

    // Determine overall status.
    $overall_status = $this->determineOverallStatus($statuses);

    $result = [
      'status' => $overall_status,
      'details' => $statuses,
    ];

    $this->memoryCache->set($cache_key, $result);
    return $result;
  }

  /**
   * Batch queries trace_mail_log for multiple mail keys at once.
   *
   * @param array<string, string> $mail_keys
   *   Array of mail keys keyed by handler ID.
   * @param int $time_start
   *   The start of the time window.
   * @param int $time_end
   *   The end of the time window.
   *
   * @return array<string, array>
   *   Status information for each handler keyed by handler ID.
   */
  protected function queryEmailStatusBatch(array $mail_keys, int $time_start, int $time_end): array {
    // Check if trace_mail_log table exists (cached).
    if (!$this->traceMailLogTableExists()) {
      $unknown_status = [
        'status' => self::STATUS_UNKNOWN,
        'message' => 'Mail log table not found',
      ];
      return array_fill_keys(array_keys($mail_keys), $unknown_status);
    }

    $statuses = [];

    // Initialize all handlers as unknown.
    foreach ($mail_keys as $handler_id => $mail_key) {
      $statuses[$handler_id] = [
        'status' => self::STATUS_UNKNOWN,
        'message' => 'No matching email log found',
      ];
    }

    // Single query for all mail keys - "sent" or "failed" events.
    $query = $this->database->select('trace_mail_log', 'tl')
      ->fields('tl', [
        'id',
        'uuid',
        'event_type',
        'status',
        'created',
        'response_code',
        'response_message',
        'mail_key',
      ])
      ->condition('tl.mail_key', array_values($mail_keys), 'IN')
      ->condition('tl.created', [$time_start, $time_end], 'BETWEEN')
      ->condition('tl.event_type', ['sent', 'failed'], 'IN')
      ->orderBy('tl.created', 'DESC');

    $results = $query->execute()->fetchAll();

    // Group results by mail_key (take the most recent for each).
    $results_by_mail_key = [];
    foreach ($results as $row) {
      if (!isset($results_by_mail_key[$row->mail_key])) {
        $results_by_mail_key[$row->mail_key] = $row;
      }
    }

    // Map results back to handler IDs.
    foreach ($mail_keys as $handler_id => $mail_key) {
      if (isset($results_by_mail_key[$mail_key])) {
        $row = $results_by_mail_key[$mail_key];
        $statuses[$handler_id] = [
          'status' => $this->validateStatus($row->status),
          'event_type' => $row->event_type,
          'response_code' => $row->response_code,
          'response_message' => $row->response_message,
          'log_id' => $row->id,
        ];
      }
    }

    // For handlers without results, check for pending "sending" events.
    $missing_mail_keys = [];
    foreach ($mail_keys as $handler_id => $mail_key) {
      if ($statuses[$handler_id]['status'] === self::STATUS_UNKNOWN) {
        $missing_mail_keys[$handler_id] = $mail_key;
      }
    }

    if (!empty($missing_mail_keys)) {
      $pending_statuses = $this->queryPendingStatusBatch($missing_mail_keys, $time_start, $time_end);
      foreach ($pending_statuses as $handler_id => $pending_status) {
        if ($pending_status['status'] !== self::STATUS_UNKNOWN) {
          $statuses[$handler_id] = $pending_status;
        }
      }
    }

    // For still-missing handlers, try fallback by time only.
    foreach ($mail_keys as $handler_id => $mail_key) {
      if ($statuses[$handler_id]['status'] === self::STATUS_UNKNOWN
        && $statuses[$handler_id]['message'] === 'No matching email log found') {
        $fallback = $this->queryEmailStatusByTimeOnly($time_start, $time_end);
        if ($fallback['status'] !== self::STATUS_UNKNOWN) {
          $statuses[$handler_id] = $fallback;
          // Only use fallback for first unknown handler to avoid duplicates.
          break;
        }
      }
    }

    return $statuses;
  }

  /**
   * Batch queries for pending "sending" events.
   *
   * @param array<string, string> $mail_keys
   *   Array of mail keys keyed by handler ID.
   * @param int $time_start
   *   The start of the time window.
   * @param int $time_end
   *   The end of the time window.
   *
   * @return array<string, array>
   *   Status information for each handler.
   */
  protected function queryPendingStatusBatch(array $mail_keys, int $time_start, int $time_end): array {
    $statuses = [];
    foreach ($mail_keys as $handler_id => $mail_key) {
      $statuses[$handler_id] = [
        'status' => self::STATUS_UNKNOWN,
        'message' => 'No matching email log found',
      ];
    }

    $query = $this->database->select('trace_mail_log', 'tl')
      ->fields('tl', ['id', 'uuid', 'event_type', 'status', 'created', 'mail_key'])
      ->condition('tl.mail_key', array_values($mail_keys), 'IN')
      ->condition('tl.created', [$time_start, $time_end], 'BETWEEN')
      ->condition('tl.event_type', 'sending')
      ->orderBy('tl.created', 'DESC');

    $results = $query->execute()->fetchAll();

    // Group by mail_key.
    $results_by_mail_key = [];
    foreach ($results as $row) {
      if (!isset($results_by_mail_key[$row->mail_key])) {
        $results_by_mail_key[$row->mail_key] = $row;
      }
    }

    // Map back to handler IDs.
    foreach ($mail_keys as $handler_id => $mail_key) {
      if (isset($results_by_mail_key[$mail_key])) {
        $row = $results_by_mail_key[$mail_key];
        $statuses[$handler_id] = [
          'status' => self::STATUS_PENDING,
          'message' => 'Email is in queue',
          'log_id' => $row->id,
        ];
      }
    }

    return $statuses;
  }

  /**
   * Checks if the trace_mail_log table exists (cached).
   *
   * @return bool
   *   TRUE if the table exists.
   */
  protected function traceMailLogTableExists(): bool {
    if ($this->tableExists === NULL) {
      $this->tableExists = $this->database->schema()->tableExists('trace_mail_log');
    }
    return $this->tableExists;
  }

  /**
   * Validates a status value against known constants.
   *
   * @param string $status
   *   The status from the database.
   *
   * @return string
   *   A valid status constant.
   */
  protected function validateStatus(string $status): string {
    $valid_statuses = [
      self::STATUS_SUCCESS,
      self::STATUS_FAILED,
      self::STATUS_PENDING,
    ];
    return in_array($status, $valid_statuses, TRUE) ? $status : self::STATUS_UNKNOWN;
  }

  /**
   * Fallback query when mail_key is not available.
   *
   * @param int $time_start
   *   The start of the time window.
   * @param int $time_end
   *   The end of the time window.
   *
   * @return array
   *   Status information.
   */
  protected function queryEmailStatusByTimeOnly(int $time_start, int $time_end): array {
    // As a fallback, look for any email in the time window.
    $query = $this->database->select('trace_mail_log', 'tl')
      ->fields('tl', ['id', 'uuid', 'event_type', 'status', 'created'])
      ->condition('tl.created', [$time_start, $time_end], 'BETWEEN')
      ->condition('tl.event_type', ['sent', 'failed'], 'IN')
      ->isNull('tl.mail_key')
      ->orderBy('tl.created', 'DESC')
      ->range(0, 1);

    $result = $query->execute()->fetchObject();

    if ($result) {
      return [
        'status' => $this->validateStatus($result->status),
        'event_type' => $result->event_type,
        'log_id' => $result->id,
        'fallback' => TRUE,
      ];
    }

    return [
      'status' => self::STATUS_UNKNOWN,
      'message' => 'No matching email log found',
    ];
  }

  /**
   * Determines the overall status from multiple handler statuses.
   *
   * @param array $statuses
   *   Array of status arrays keyed by handler ID.
   *
   * @return string
   *   The overall status.
   */
  protected function determineOverallStatus(array $statuses): string {
    $has_failed = FALSE;
    $has_pending = FALSE;
    $has_success = FALSE;

    foreach ($statuses as $status_info) {
      $status = $status_info['status'];

      switch ($status) {
        case self::STATUS_FAILED:
          $has_failed = TRUE;
          break;

        case self::STATUS_PENDING:
          $has_pending = TRUE;
          break;

        case self::STATUS_SUCCESS:
          $has_success = TRUE;
          break;
      }
    }

    // Priority: failed > pending > success > unknown.
    if ($has_failed) {
      return self::STATUS_FAILED;
    }
    if ($has_pending) {
      return self::STATUS_PENDING;
    }
    if ($has_success) {
      return self::STATUS_SUCCESS;
    }
    return self::STATUS_UNKNOWN;
  }

  /**
   * Batch loads email statuses for multiple submissions.
   *
   * @param \Drupal\webform\WebformSubmissionInterface[] $submissions
   *   Array of webform submissions.
   *
   * @return array<int, array>
   *   Array of status information keyed by submission ID.
   */
  public function getEmailStatusesBatch(array $submissions): array {
    $results = [];

    foreach ($submissions as $submission) {
      $results[$submission->id()] = $this->getEmailStatus($submission);
    }

    return $results;
  }

  /**
   * Clears the memory cache.
   */
  public function clearCache(): void {
    $this->memoryCache->deleteAll();
    $this->tableExists = NULL;
  }

}
