<?php

declare(strict_types=1);

namespace Drupal\trace_mail_log\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Datetime\DateFormatterInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\StreamedResponse;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;

/**
 * Controller for exporting mail log entries.
 */
class ExportController extends ControllerBase {

  /**
   * The database connection.
   */
  protected Connection $database;

  /**
   * The date formatter service.
   */
  protected DateFormatterInterface $dateFormatter;

  /**
   * Constructs an ExportController.
   */
  public function __construct(Connection $database, DateFormatterInterface $date_formatter) {
    $this->database = $database;
    $this->dateFormatter = $date_formatter;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('database'),
      $container->get('date.formatter'),
    );
  }

  /**
   * Exports filtered log entries.
   *
   * @param string $format
   *   Export format: csv or json.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The export response.
   */
  public function exportFiltered(string $format, Request $request): Response {
    $this->validateFormat($format);

    // Build query with filters.
    $query = $this->buildQuery($request);
    $results = $query->execute()->fetchAll();

    return $this->generateExport($results, $format);
  }

  /**
   * Exports all log entries.
   *
   * @param string $format
   *   Export format: csv or json.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The export response.
   */
  public function exportAll(string $format): Response {
    $this->validateFormat($format);

    // Get all entries without filters.
    $query = $this->database->select('trace_mail_log', 'm')
      ->fields('m')
      ->orderBy('created', 'DESC');

    $results = $query->execute()->fetchAll();

    return $this->generateExport($results, $format);
  }

  /**
   * Validates the export format.
   *
   * @param string $format
   *   The format to validate.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\BadRequestHttpException
   *   If format is invalid.
   */
  protected function validateFormat(string $format): void {
    if (!in_array($format, ['csv', 'json'], TRUE)) {
      throw new BadRequestHttpException('Invalid export format. Use csv or json.');
    }
  }

  /**
   * Builds a query with filters from request.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Drupal\Core\Database\Query\SelectInterface
   *   The query object.
   */
  protected function buildQuery(Request $request) {
    $query = $this->database->select('trace_mail_log', 'm')
      ->fields('m')
      ->orderBy('created', 'DESC');

    // Apply filters.
    if ($status = $request->query->get('status')) {
      $query->condition('status', $status);
    }
    if ($event_type = $request->query->get('event_type')) {
      $query->condition('event_type', $event_type);
    }
    if ($recipient = $request->query->get('recipient')) {
      $query->condition('recipients', '%' . $this->database->escapeLike($recipient) . '%', 'LIKE');
    }
    if ($subject = $request->query->get('subject')) {
      $query->condition('subject', '%' . $this->database->escapeLike($subject) . '%', 'LIKE');
    }

    return $query;
  }

  /**
   * Generates the export response.
   *
   * @param array $results
   *   The database results.
   * @param string $format
   *   The export format.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The response object.
   */
  protected function generateExport(array $results, string $format): Response {
    $filename = 'mail-log-export-' . date('Y-m-d') . '.' . $format;

    if ($format === 'csv') {
      return $this->generateCsv($results, $filename);
    }

    return $this->generateJson($results, $filename);
  }

  /**
   * Generates a CSV export.
   *
   * @param array $results
   *   The database results.
   * @param string $filename
   *   The download filename.
   *
   * @return \Symfony\Component\HttpFoundation\StreamedResponse
   *   The streamed response.
   */
  protected function generateCsv(array $results, string $filename): StreamedResponse {
    $response = new StreamedResponse(function () use ($results) {
      $handle = fopen('php://output', 'w');

      // Write CSV header.
      fputcsv($handle, [
        'ID',
        'UUID',
        'Message ID',
        'Event Type',
        'Status',
        'Mail Key',
        'Sender',
        'To',
        'Cc',
        'Bcc',
        'Subject',
        'Transport Type',
        'Response Code',
        'Response Message',
        'Created',
      ]);

      // Write data rows.
      foreach ($results as $row) {
        $recipients = json_decode($row->recipients, TRUE) ?? [];
        $to = implode('; ', $recipients['to'] ?? []);
        $cc = implode('; ', $recipients['cc'] ?? []);
        $bcc = implode('; ', $recipients['bcc'] ?? []);

        fputcsv($handle, [
          $row->id,
          $row->uuid,
          $row->message_id,
          $row->event_type,
          $row->status,
          $row->mail_key,
          $row->sender,
          $to,
          $cc,
          $bcc,
          $row->subject,
          $row->transport_type,
          $row->response_code,
          $row->response_message,
          $this->dateFormatter->format((int) $row->created, 'custom', 'Y-m-d H:i:s'),
        ]);
      }

      fclose($handle);
    });

    $response->headers->set('Content-Type', 'text/csv; charset=utf-8');
    $response->headers->set('Content-Disposition', 'attachment; filename="' . $filename . '"');

    return $response;
  }

  /**
   * Generates a JSON export.
   *
   * @param array $results
   *   The database results.
   * @param string $filename
   *   The download filename.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The response object.
   */
  protected function generateJson(array $results, string $filename): Response {
    $data = [];

    foreach ($results as $row) {
      $recipients = json_decode($row->recipients, TRUE) ?? [];

      $data[] = [
        'id' => (int) $row->id,
        'uuid' => $row->uuid,
        'message_id' => $row->message_id,
        'event_type' => $row->event_type,
        'status' => $row->status,
        'mail_key' => $row->mail_key,
        'sender' => $row->sender,
        'recipients' => [
          'to' => $recipients['to'] ?? [],
          'cc' => $recipients['cc'] ?? [],
          'bcc' => $recipients['bcc'] ?? [],
        ],
        'subject' => $row->subject,
        'transport_type' => $row->transport_type,
        'response_code' => $row->response_code ? (int) $row->response_code : NULL,
        'response_message' => $row->response_message,
        'created' => $this->dateFormatter->format((int) $row->created, 'custom', 'Y-m-d\TH:i:sP'),
        'created_timestamp' => (int) $row->created,
      ];
    }

    $json = json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);

    $response = new Response($json);
    $response->headers->set('Content-Type', 'application/json; charset=utf-8');
    $response->headers->set('Content-Disposition', 'attachment; filename="' . $filename . '"');

    return $response;
  }

}
