<?php

declare(strict_types=1);

namespace Drupal\Tests\trace_mail_log\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\Tests\trace_mail_log\Traits\MailLogTestTrait;
use Drupal\trace_mail_log\Service\PurgeService;

/**
 * Tests the PurgeService.
 *
 * @coversDefaultClass \Drupal\trace_mail_log\Service\PurgeService
 * @group trace_mail_log
 */
class PurgeServiceTest extends KernelTestBase {

  use MailLogTestTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['trace_mail_log', 'symfony_mailer'];

  /**
   * The purge service.
   *
   * @var \Drupal\trace_mail_log\Service\PurgeService
   */
  protected PurgeService $purgeService;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installSchema('trace_mail_log', ['trace_mail_log']);
    $this->installConfig(['trace_mail_log']);

    $this->purgeService = $this->container->get('trace_mail_log.purge_service');
  }

  /**
   * Tests that the service is available.
   */
  public function testServiceExists(): void {
    $this->assertInstanceOf(PurgeService::class, $this->purgeService);
  }

  /**
   * Tests purge removes old entries.
   *
   * @covers ::purge
   */
  public function testPurgeRemovesOldEntries(): void {
    // Create an old entry (40 days ago).
    $old_time = strtotime('-40 days');
    $this->createMailLogEntry(['created' => $old_time]);

    // Create a recent entry.
    $this->createMailLogEntry(['created' => time()]);

    // Default retention is 30 days.
    $purged = $this->purgeService->purge();

    $this->assertEquals(1, $purged);
    $this->assertEquals(1, $this->countMailLogEntries());
  }

  /**
   * Tests purge respects retention days setting.
   *
   * @covers ::purge
   */
  public function testPurgeRespectsRetentionDays(): void {
    // Set retention to 7 days.
    $config = $this->config('trace_mail_log.settings');
    $config->set('retention_days', 7)->save();

    // Create entry 10 days ago (should be purged).
    $this->createMailLogEntry(['created' => strtotime('-10 days')]);

    // Create entry 5 days ago (should remain).
    $this->createMailLogEntry(['created' => strtotime('-5 days')]);

    $purged = $this->purgeService->purge();

    $this->assertEquals(1, $purged);
    $this->assertEquals(1, $this->countMailLogEntries());
  }

  /**
   * Tests purge returns zero when retention is disabled.
   *
   * @covers ::purge
   */
  public function testPurgeDisabledWhenRetentionZero(): void {
    $config = $this->config('trace_mail_log.settings');
    $config->set('retention_days', 0)->save();

    // Create old entries.
    $this->createMailLogEntry(['created' => strtotime('-100 days')]);
    $this->createMailLogEntry(['created' => strtotime('-200 days')]);

    $purged = $this->purgeService->purge();

    $this->assertEquals(0, $purged);
    $this->assertEquals(2, $this->countMailLogEntries());
  }

  /**
   * Tests purge with no entries to purge.
   *
   * @covers ::purge
   */
  public function testPurgeWithNoOldEntries(): void {
    // Create only recent entries.
    $this->createMailLogEntries(5, ['created' => time()]);

    $purged = $this->purgeService->purge();

    $this->assertEquals(0, $purged);
    $this->assertEquals(5, $this->countMailLogEntries());
  }

  /**
   * Tests deleteSingle removes specific entry.
   *
   * @covers ::deleteSingle
   */
  public function testDeleteSingle(): void {
    $id1 = $this->createMailLogEntry(['subject' => 'Keep this']);
    $id2 = $this->createMailLogEntry(['subject' => 'Delete this']);

    $result = $this->purgeService->deleteSingle($id2);

    $this->assertTrue($result);
    $this->assertEquals(1, $this->countMailLogEntries());
    $this->assertNotNull($this->getMailLogEntry($id1));
    $this->assertNull($this->getMailLogEntry($id2));
  }

  /**
   * Tests deleteSingle returns false for non-existent entry.
   *
   * @covers ::deleteSingle
   */
  public function testDeleteSingleNonExistent(): void {
    $result = $this->purgeService->deleteSingle(99999);

    $this->assertFalse($result);
  }

  /**
   * Tests deleteAll removes all entries.
   *
   * @covers ::deleteAll
   */
  public function testDeleteAll(): void {
    $this->createMailLogEntries(10);

    $deleted = $this->purgeService->deleteAll();

    $this->assertEquals(10, $deleted);
    $this->assertEquals(0, $this->countMailLogEntries());
  }

  /**
   * Tests deleteAll returns zero when no entries.
   *
   * @covers ::deleteAll
   */
  public function testDeleteAllEmpty(): void {
    $deleted = $this->purgeService->deleteAll();

    $this->assertEquals(0, $deleted);
  }

  /**
   * Tests getStats returns correct counts.
   *
   * @covers ::getStats
   */
  public function testGetStats(): void {
    // Create various entries.
    $now = time();
    $today_start = strtotime('today');
    $week_ago = strtotime('-7 days');

    // Sent today.
    $this->createMailLogEntry([
      'event_type' => 'sent',
      'status' => 'success',
      'created' => $now,
    ]);
    $this->createMailLogEntry([
      'event_type' => 'sent',
      'status' => 'success',
      'created' => $now - 3600,
    ]);

    // Sent this week (but not today).
    $this->createMailLogEntry([
      'event_type' => 'sent',
      'status' => 'success',
      'created' => strtotime('-3 days'),
    ]);

    // Failed this week.
    $this->createFailedMailLogEntry([
      'created' => strtotime('-2 days'),
    ]);

    // Old entry (more than a week).
    $this->createMailLogEntry([
      'event_type' => 'sent',
      'status' => 'success',
      'created' => strtotime('-10 days'),
    ]);

    $stats = $this->purgeService->getStats();

    $this->assertEquals(5, $stats['total']);
    $this->assertEquals(2, $stats['today']);
    $this->assertEquals(3, $stats['this_week']);
    $this->assertEquals(1, $stats['failures_this_week']);
  }

  /**
   * Tests getStats with empty database.
   *
   * @covers ::getStats
   */
  public function testGetStatsEmpty(): void {
    $stats = $this->purgeService->getStats();

    $this->assertEquals(0, $stats['total']);
    $this->assertEquals(0, $stats['today']);
    $this->assertEquals(0, $stats['this_week']);
    $this->assertEquals(0, $stats['failures_this_week']);
  }

  /**
   * Tests getStats only counts sent events for today/week.
   *
   * @covers ::getStats
   */
  public function testGetStatsOnlyCountsSentEvents(): void {
    $now = time();

    // Queued event should not count for today/week stats.
    $this->createQueuedMailLogEntry(['created' => $now]);

    // Failed event should only count in failures.
    $this->createFailedMailLogEntry(['created' => $now]);

    // Sent event should count.
    $this->createMailLogEntry([
      'event_type' => 'sent',
      'status' => 'success',
      'created' => $now,
    ]);

    $stats = $this->purgeService->getStats();

    $this->assertEquals(3, $stats['total']);
    $this->assertEquals(1, $stats['today']);
    $this->assertEquals(1, $stats['this_week']);
    $this->assertEquals(1, $stats['failures_this_week']);
  }

}
