<?php

declare(strict_types=1);

namespace Drupal\trace_mail_log\Form;

use Drupal\trace_mail_log\Service\PurgeService;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Confirmation form for deleting all mail log entries.
 */
class DeleteAllForm extends ConfirmFormBase {

  /**
   * The purge service.
   */
  protected PurgeService $purgeService;

  /**
   * Constructs a DeleteAllForm.
   */
  public function __construct(PurgeService $purge_service) {
    $this->purgeService = $purge_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('trace_mail_log.purge_service'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'trace_mail_log_delete_all_form';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Are you sure you want to delete ALL mail log entries?');
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    $stats = $this->purgeService->getStats();
    return $this->t('This will permanently delete <strong>@count</strong> log entries and all associated transcript files. This action cannot be undone.', [
      '@count' => $stats['total'],
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl(): Url {
    return new Url('trace_mail_log.list');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText() {
    return $this->t('Delete all entries');
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $stats = $this->purgeService->getStats();

    if ($stats['total'] === 0) {
      $this->messenger()->addWarning($this->t('There are no mail log entries to delete.'));
      return $this->redirect('trace_mail_log.list')->send();
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $count = $this->purgeService->deleteAll();

    if ($count > 0) {
      $this->messenger()->addStatus($this->t('Successfully deleted @count mail log entries.', ['@count' => $count]));
    }
    else {
      $this->messenger()->addWarning($this->t('No entries were deleted.'));
    }

    $form_state->setRedirectUrl($this->getCancelUrl());
  }

}
