<?php

declare(strict_types=1);

namespace Drupal\Tests\trace_mail_log\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\Tests\trace_mail_log\Traits\MailLogTestTrait;
use Drupal\trace_mail_log\Service\MailLogService;
use Symfony\Component\Mime\Address;
use Symfony\Component\Mime\Email;

/**
 * Tests the MailLogService.
 *
 * @coversDefaultClass \Drupal\trace_mail_log\Service\MailLogService
 * @group trace_mail_log
 */
class MailLogServiceTest extends KernelTestBase {

  use MailLogTestTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['trace_mail_log', 'symfony_mailer'];

  /**
   * The mail log service.
   *
   * @var \Drupal\trace_mail_log\Service\MailLogService
   */
  protected MailLogService $mailLogService;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installSchema('trace_mail_log', ['trace_mail_log']);
    $this->installConfig(['trace_mail_log']);

    $this->mailLogService = $this->container->get('trace_mail_log.mail_log_service');
  }

  /**
   * Tests that the service is available.
   */
  public function testServiceExists(): void {
    $this->assertInstanceOf(MailLogService::class, $this->mailLogService);
  }

  /**
   * Tests isEnabled returns true by default.
   *
   * @covers ::isEnabled
   */
  public function testIsEnabledDefault(): void {
    $this->assertTrue($this->mailLogService->isEnabled());
  }

  /**
   * Tests isEnabled returns false when disabled.
   *
   * @covers ::isEnabled
   */
  public function testIsEnabledWhenDisabled(): void {
    $config = $this->config('trace_mail_log.settings');
    $config->set('enabled', FALSE)->save();

    $this->assertFalse($this->mailLogService->isEnabled());
  }

  /**
   * Tests generateUuid returns valid UUID.
   *
   * @covers ::generateUuid
   */
  public function testGenerateUuid(): void {
    $uuid = $this->mailLogService->generateUuid();

    $this->assertNotEmpty($uuid);
    $this->assertMatchesRegularExpression(
      '/^[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}$/i',
      $uuid
    );
  }

  /**
   * Tests generateUuid returns unique values.
   *
   * @covers ::generateUuid
   */
  public function testGenerateUuidUnique(): void {
    $uuid1 = $this->mailLogService->generateUuid();
    $uuid2 = $this->mailLogService->generateUuid();

    $this->assertNotEquals($uuid1, $uuid2);
  }

  /**
   * Tests log creates database entry.
   *
   * @covers ::log
   */
  public function testLogCreatesEntry(): void {
    $uuid = $this->mailLogService->generateUuid();

    $this->mailLogService->log($uuid, 'sent', 'success', [
      'mail_key' => 'test_module_test',
      'sender' => 'sender@example.com',
      'recipients' => ['to' => ['recipient@example.com']],
      'subject' => 'Test Subject',
      'transport_type' => 'smtp',
      'response_code' => 250,
      'response_message' => 'OK',
    ]);

    $count = $this->countMailLogEntries(['uuid' => $uuid]);
    $this->assertEquals(1, $count);
  }

  /**
   * Tests log does not create entry when disabled.
   *
   * @covers ::log
   */
  public function testLogDoesNotCreateWhenDisabled(): void {
    $config = $this->config('trace_mail_log.settings');
    $config->set('enabled', FALSE)->save();

    $uuid = $this->mailLogService->generateUuid();
    $this->mailLogService->log($uuid, 'sent', 'success', [
      'subject' => 'Test Subject',
    ]);

    $count = $this->countMailLogEntries(['uuid' => $uuid]);
    $this->assertEquals(0, $count);
  }

  /**
   * Tests log stores all fields correctly.
   *
   * @covers ::log
   */
  public function testLogStoresAllFields(): void {
    $uuid = $this->mailLogService->generateUuid();
    $message_id = '<test123@example.com>';

    $this->mailLogService->log($uuid, 'sent', 'success', [
      'message_id' => $message_id,
      'mail_key' => 'contact_mail',
      'sender' => 'sender@example.com',
      'recipients' => [
        'to' => ['to@example.com'],
        'cc' => ['cc@example.com'],
        'bcc' => [],
      ],
      'subject' => 'Test Email Subject',
      'transport_type' => 'smtp',
      'response_code' => 250,
      'response_message' => '2.0.0 Ok: queued',
    ]);

    $database = \Drupal::database();
    $entry = $database->select('trace_mail_log', 'm')
      ->fields('m')
      ->condition('uuid', $uuid)
      ->execute()
      ->fetchObject();

    $this->assertNotNull($entry);
    $this->assertEquals($uuid, $entry->uuid);
    $this->assertEquals($message_id, $entry->message_id);
    $this->assertEquals('sent', $entry->event_type);
    $this->assertEquals('success', $entry->status);
    $this->assertEquals('contact_mail', $entry->mail_key);
    $this->assertEquals('sender@example.com', $entry->sender);
    $this->assertEquals('Test Email Subject', $entry->subject);
    $this->assertEquals('smtp', $entry->transport_type);
    $this->assertEquals(250, $entry->response_code);
    $this->assertEquals('2.0.0 Ok: queued', $entry->response_message);

    // Check recipients JSON.
    $recipients = json_decode($entry->recipients, TRUE);
    $this->assertEquals(['to@example.com'], $recipients['to']);
    $this->assertEquals(['cc@example.com'], $recipients['cc']);
  }

  /**
   * Tests log truncates long subjects.
   *
   * @covers ::log
   */
  public function testLogTruncatesLongSubject(): void {
    $uuid = $this->mailLogService->generateUuid();
    $long_subject = str_repeat('A', 300);

    $this->mailLogService->log($uuid, 'sent', 'success', [
      'subject' => $long_subject,
    ]);

    $database = \Drupal::database();
    $entry = $database->select('trace_mail_log', 'm')
      ->fields('m', ['subject'])
      ->condition('uuid', $uuid)
      ->execute()
      ->fetchObject();

    $this->assertEquals(255, strlen($entry->subject));
  }

  /**
   * Tests log with different event types.
   *
   * @covers ::log
   * @dataProvider eventTypeProvider
   */
  public function testLogEventTypes(string $event_type, string $status): void {
    $uuid = $this->mailLogService->generateUuid();

    $this->mailLogService->log($uuid, $event_type, $status, [
      'subject' => 'Test',
    ]);

    $database = \Drupal::database();
    $entry = $database->select('trace_mail_log', 'm')
      ->fields('m', ['event_type', 'status'])
      ->condition('uuid', $uuid)
      ->execute()
      ->fetchObject();

    $this->assertEquals($event_type, $entry->event_type);
    $this->assertEquals($status, $entry->status);
  }

  /**
   * Data provider for event types.
   *
   * @return array
   *   Test cases.
   */
  public static function eventTypeProvider(): array {
    return [
      'queued' => ['queued', 'pending'],
      'sending' => ['sending', 'pending'],
      'sent' => ['sent', 'success'],
      'failed' => ['failed', 'failed'],
      'requeued' => ['requeued', 'pending'],
    ];
  }

  /**
   * Tests extractEmailData extracts all fields from Email object.
   *
   * @covers ::extractEmailData
   */
  public function testExtractEmailData(): void {
    $email = (new Email())
      ->from(new Address('sender@example.com', 'Sender Name'))
      ->to(new Address('to@example.com', 'Recipient'))
      ->cc('cc@example.com')
      ->subject('Test Subject')
      ->text('Plain text body')
      ->html('<p>HTML body</p>');

    $data = $this->mailLogService->extractEmailData($email, 'smtp://localhost');

    $this->assertEquals('Test Subject', $data['subject']);
    $this->assertEquals('sender@example.com', $data['sender']);
    $this->assertEquals(['to@example.com'], $data['recipients']['to']);
    $this->assertEquals(['cc@example.com'], $data['recipients']['cc']);
    $this->assertEquals([], $data['recipients']['bcc']);
    $this->assertEquals('Plain text body', $data['body_text']);
    $this->assertEquals('<p>HTML body</p>', $data['body_html']);
    $this->assertEquals('smtp', $data['transport_type']);
  }

  /**
   * Tests extractEmailData with minimal email.
   *
   * @covers ::extractEmailData
   */
  public function testExtractEmailDataMinimal(): void {
    $email = (new Email())
      ->to('to@example.com')
      ->subject('Minimal');

    $data = $this->mailLogService->extractEmailData($email);

    $this->assertEquals('Minimal', $data['subject']);
    $this->assertNull($data['sender']);
    $this->assertEquals(['to@example.com'], $data['recipients']['to']);
  }

}
