<?php

declare(strict_types=1);

namespace Drupal\Tests\trace_mail_log\Traits;

use Drupal\Component\Uuid\Php as PhpUuid;

/**
 * Provides helper methods for mail log tests.
 */
trait MailLogTestTrait {

  /**
   * Creates a mail log entry in the database.
   *
   * @param array $values
   *   Optional values to override defaults.
   *
   * @return int
   *   The ID of the created entry.
   */
  protected function createMailLogEntry(array $values = []): int {
    $uuid = new PhpUuid();

    $defaults = [
      'uuid' => $uuid->generate(),
      'message_id' => '<' . $uuid->generate() . '@example.com>',
      'event_type' => 'sent',
      'status' => 'success',
      'mail_key' => 'test_module_test_key',
      'sender' => 'sender@example.com',
      'recipients' => json_encode([
        'to' => ['recipient@example.com'],
        'cc' => [],
        'bcc' => [],
      ]),
      'subject' => 'Test email subject',
      'transport_type' => 'smtp',
      'response_code' => 250,
      'response_message' => 'OK',
      'transcript_file' => NULL,
      'created' => time(),
    ];

    $fields = array_merge($defaults, $values);

    // Ensure recipients is JSON encoded if passed as array.
    if (isset($values['recipients']) && is_array($values['recipients'])) {
      $fields['recipients'] = json_encode($values['recipients']);
    }

    $database = \Drupal::database();
    return (int) $database->insert('trace_mail_log')
      ->fields($fields)
      ->execute();
  }

  /**
   * Creates multiple mail log entries.
   *
   * @param int $count
   *   Number of entries to create.
   * @param array $values
   *   Optional values to override defaults for all entries.
   *
   * @return array
   *   Array of created entry IDs.
   */
  protected function createMailLogEntries(int $count, array $values = []): array {
    $ids = [];
    for ($i = 0; $i < $count; $i++) {
      $entry_values = $values;
      // Make subject unique if not specified.
      if (!isset($values['subject'])) {
        $entry_values['subject'] = "Test email subject #{$i}";
      }
      // Vary created time if not specified (for sorting tests).
      if (!isset($values['created'])) {
        $entry_values['created'] = time() - ($count - $i) * 60;
      }
      $ids[] = $this->createMailLogEntry($entry_values);
    }
    return $ids;
  }

  /**
   * Creates a failed mail log entry.
   *
   * @param array $values
   *   Optional values to override defaults.
   *
   * @return int
   *   The ID of the created entry.
   */
  protected function createFailedMailLogEntry(array $values = []): int {
    $defaults = [
      'event_type' => 'failed',
      'status' => 'failed',
      'response_code' => 550,
      'response_message' => 'Mailbox not found',
    ];
    return $this->createMailLogEntry(array_merge($defaults, $values));
  }

  /**
   * Creates a queued mail log entry.
   *
   * @param array $values
   *   Optional values to override defaults.
   *
   * @return int
   *   The ID of the created entry.
   */
  protected function createQueuedMailLogEntry(array $values = []): int {
    $defaults = [
      'event_type' => 'queued',
      'status' => 'pending',
      'response_code' => NULL,
      'response_message' => NULL,
    ];
    return $this->createMailLogEntry(array_merge($defaults, $values));
  }

  /**
   * Gets a mail log entry by ID.
   *
   * @param int $id
   *   The entry ID.
   *
   * @return object|null
   *   The entry object or NULL if not found.
   */
  protected function getMailLogEntry(int $id): ?object {
    $database = \Drupal::database();
    $result = $database->select('trace_mail_log', 'm')
      ->fields('m')
      ->condition('id', $id)
      ->execute()
      ->fetchObject();

    return $result ?: NULL;
  }

  /**
   * Counts mail log entries.
   *
   * @param array $conditions
   *   Optional conditions as field => value pairs.
   *
   * @return int
   *   The count of entries.
   */
  protected function countMailLogEntries(array $conditions = []): int {
    $database = \Drupal::database();
    $query = $database->select('trace_mail_log', 'm');

    foreach ($conditions as $field => $value) {
      $query->condition($field, $value);
    }

    return (int) $query->countQuery()->execute()->fetchField();
  }

}
