<?php

declare(strict_types=1);

namespace Drupal\trace_mail_log\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Settings form for mail log configuration.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * Constructs a SettingsForm.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typedConfigManager
   *   The typed config manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $dateFormatter
   *   The date formatter service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typedConfigManager,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly DateFormatterInterface $dateFormatter,
    protected readonly ModuleHandlerInterface $moduleHandler,
  ) {
    parent::__construct($config_factory, $typedConfigManager);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('entity_type.manager'),
      $container->get('date.formatter'),
      $container->get('module_handler'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'trace_mail_log_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['trace_mail_log.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable mail logging'),
      '#description' => $this->t('When enabled, all email operations will be logged.'),
      '#config_target' => 'trace_mail_log.settings:enabled',
    ];

    $form['retention_days'] = [
      '#type' => 'number',
      '#title' => $this->t('Retention period (days)'),
      '#description' => $this->t('Log entries older than this will be automatically purged. Set to 0 to disable automatic purging.'),
      '#config_target' => 'trace_mail_log.settings:retention_days',
      '#min' => 0,
      '#max' => 365,
    ];

    $form['log_directory'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Log file directory'),
      '#description' => $this->t('Directory where transcript files are stored. Use private:// for security.'),
      '#config_target' => 'trace_mail_log.settings:log_directory',
    ];

    $form['log_body'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Log email body'),
      '#description' => $this->t('Include email HTML and text body in log files. Disable for privacy.'),
      '#config_target' => 'trace_mail_log.settings:log_body',
    ];

    $form['log_attachments'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Log attachments'),
      '#description' => $this->t('Include attachment information in logs. Note: Attachment content is not stored.'),
      '#config_target' => 'trace_mail_log.settings:log_attachments',
    ];

    // Display settings.
    $form['display'] = [
      '#type' => 'details',
      '#title' => $this->t('Display settings'),
      '#open' => TRUE,
    ];

    // Build date format options from available Drupal date formats.
    $date_formats = $this->entityTypeManager->getStorage('date_format')->loadMultiple();
    $date_format_options = [];
    $example_timestamp = time();
    foreach ($date_formats as $id => $format) {
      $example = $this->dateFormatter->format($example_timestamp, $id);
      $date_format_options[$id] = $format->label() . ' (' . $example . ')';
    }

    $form['display']['date_format_list'] = [
      '#type' => 'select',
      '#title' => $this->t('Date format for list view'),
      '#description' => $this->t('Select the date format to use in the mail log list.'),
      '#options' => $date_format_options,
      '#config_target' => 'trace_mail_log.settings:date_format_list',
    ];

    $form['display']['date_format_detail'] = [
      '#type' => 'select',
      '#title' => $this->t('Date format for detail view'),
      '#description' => $this->t('Select the date format to use in the mail log detail page.'),
      '#options' => $date_format_options,
      '#config_target' => 'trace_mail_log.settings:date_format_detail',
    ];

    $form['display']['items_per_page'] = [
      '#type' => 'number',
      '#title' => $this->t('Default items per page'),
      '#description' => $this->t('Default number of log entries to display per page. Users can override this temporarily.'),
      '#config_target' => 'trace_mail_log.settings:items_per_page',
      '#min' => 10,
      '#max' => 500,
    ];

    // Webform integration settings (only visible when webform is installed).
    if ($this->moduleHandler->moduleExists('webform')) {
      $form['webform'] = [
        '#type' => 'details',
        '#title' => $this->t('Webform integration'),
        '#open' => TRUE,
      ];

      $form['webform']['webform_integration'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Show email status column in webform submissions'),
        '#description' => $this->t('When enabled, an "Email" status column will be added to webform submission tables showing the delivery status of emails sent by email handlers.'),
        '#config_target' => 'trace_mail_log.settings:webform_integration',
      ];
    }

    return parent::buildForm($form, $form_state);
  }

}
