<?php

declare(strict_types=1);

namespace Drupal\trace_mail_log\Hook;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\trace_mail_log\Service\PurgeService;

/**
 * Hook implementations for the Trace Mail Log module.
 */
class TraceMailLogHooks {

  /**
   * Constructs a TraceMailLogHooks object.
   *
   * @param \Drupal\trace_mail_log\Service\PurgeService $purgeService
   *   The purge service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   */
  public function __construct(
    protected readonly PurgeService $purgeService,
    protected readonly ConfigFactoryInterface $configFactory,
  ) {}

  /**
   * Implements hook_theme().
   */
  #[Hook('theme')]
  public function theme(): array {
    return [
      'mail_log_dashboard' => [
        'variables' => [
          'stats' => [],
          'recent_failures' => [],
          'list_url' => NULL,
        ],
        'template' => 'mail-log-dashboard',
      ],
      'mail_log_list' => [
        'variables' => [
          'rows' => [],
          'filters' => [],
          'pager' => NULL,
          'can_delete' => FALSE,
          'sort' => 'created',
          'order' => 'desc',
          'sort_links' => [],
          'items_per_page' => 50,
          'allowed_limits' => [],
          'limit_links' => [],
          'export_urls' => [],
        ],
        'template' => 'mail-log-list',
      ],
      'mail_log_detail' => [
        'variables' => [
          'entry' => [],
          'transcript_content' => NULL,
          'timeline' => [],
          'download_url' => NULL,
          'can_delete' => FALSE,
        ],
        'template' => 'mail-log-detail',
      ],
    ];
  }

  /**
   * Implements hook_preprocess_HOOK() for mail_log_dashboard.
   */
  #[Hook('preprocess_mail_log_dashboard')]
  public function preprocessMailLogDashboard(array &$variables): void {
    $variables['#attached']['library'][] = 'trace_mail_log/mail-log';
  }

  /**
   * Implements hook_preprocess_HOOK() for mail_log_list.
   */
  #[Hook('preprocess_mail_log_list')]
  public function preprocessMailLogList(array &$variables): void {
    $variables['#attached']['library'][] = 'trace_mail_log/mail-log';
  }

  /**
   * Implements hook_preprocess_HOOK() for mail_log_detail.
   */
  #[Hook('preprocess_mail_log_detail')]
  public function preprocessMailLogDetail(array &$variables): void {
    $variables['#attached']['library'][] = 'trace_mail_log/mail-log';
  }

  /**
   * Implements hook_cron().
   */
  #[Hook('cron')]
  public function cron(): void {
    $this->purgeService->purge();
  }

  /**
   * Implements hook_mail_alter().
   *
   * Adds the X-Drupal-Mail-Key header to all emails for tracking.
   */
  #[Hook('mail_alter')]
  public function mailAlter(array &$message): void {
    if (!isset($message['module'], $message['key'])) {
      return;
    }

    $mail_key = $message['module'] . '_' . $message['key'];
    $message['headers']['X-Drupal-Mail-Key'] = $mail_key;
  }

  /**
   * Implements hook_entity_type_alter().
   *
   * Adds custom list builder handler for webform submissions when integration
   * is enabled.
   *
   * @param array<string, \Drupal\Core\Entity\EntityTypeInterface> $entity_types
   *   An associative array of all entity type definitions.
   */
  #[Hook('entity_type_alter')]
  public function entityTypeAlter(array &$entity_types): void {
    if (!isset($entity_types['webform_submission'])) {
      return;
    }

    $config = $this->configFactory->get('trace_mail_log.settings');
    if (!$config->get('webform_integration')) {
      return;
    }

    /** @var \Drupal\Core\Entity\EntityTypeInterface $webform_submission_type */
    $webform_submission_type = $entity_types['webform_submission'];
    $webform_submission_type->setListBuilderClass('Drupal\trace_mail_log\WebformSubmissionEmailStatusListBuilder');
  }

}
