<?php

declare(strict_types=1);

namespace Drupal\trace_mail_log;

use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Render\RenderContext;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\Core\StringTranslation\TranslationManager;
use Drupal\Core\Url;
use Drupal\symfony_mailer\Exception\MissingTransportException;
use Drupal\symfony_mailer\Exception\SkipMailException;
use Drupal\symfony_mailer\InternalEmailInterface;
use Drupal\symfony_mailer\MailerInterface;
use Drupal\user\Entity\User;
use Symfony\Component\Mailer\Mailer as SymfonyMailer;
use Symfony\Component\Mailer\Transport;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;

/**
 * Decorates symfony_mailer to inject event dispatcher into transport.
 *
 * The default symfony_mailer module creates transports without injecting the
 * event dispatcher, which means Symfony Mailer events (MessageEvent,
 * SentMessageEvent, FailedMessageEvent) are never dispatched.
 *
 * This decorator intercepts the send process and ensures the event dispatcher
 * is properly injected into the transport, enabling event subscribers to
 * receive mail events.
 *
 * @see https://www.drupal.org/project/symfony_mailer_lite/issues/3418269
 */
class MailerDecorator implements MailerInterface {

  use StringTranslationTrait;

  /**
   * Constructs a MailerDecorator.
   *
   * @param \Drupal\symfony_mailer\MailerInterface $innerMailer
   *   The decorated mailer service.
   * @param \Symfony\Contracts\EventDispatcher\EventDispatcherInterface $eventDispatcher
   *   The event dispatcher.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $stringTranslation
   *   The string translation service.
   */
  public function __construct(
    protected readonly MailerInterface $innerMailer,
    protected readonly EventDispatcherInterface $eventDispatcher,
    protected readonly MessengerInterface $messenger,
    TranslationInterface $stringTranslation,
  ) {
    $this->setStringTranslation($stringTranslation);
  }

  /**
   * {@inheritdoc}
   */
  public function send(InternalEmailInterface $email): bool {
    // We need to intercept after the email is fully processed but before
    // the actual SMTP send. The inner mailer's send() method calls doSend()
    // which creates the transport without the event dispatcher.
    //
    // Strategy: We call the inner mailer's send() which handles all the
    // complex logic (rendering, language switching, theme switching, etc.)
    // but we've already registered our event subscribers. The problem is
    // the transport doesn't dispatch events.
    //
    // Since we can't easily intercept doSend(), we use a different approach:
    // We let the original mailer do its work, but the events won't fire.
    // Instead, we need to override the actual sending mechanism.
    //
    // Actually, we need to completely replace the send logic to inject
    // the dispatcher into the transport.
    return $this->doSendWithEvents($email);
  }

  /**
   * Sends an email with event dispatcher properly injected.
   *
   * This reimplements the send logic from the original Mailer class,
   * but ensures Transport::fromDsn() is called with the event dispatcher.
   *
   * @param \Drupal\symfony_mailer\InternalEmailInterface $email
   *   The email to send.
   *
   * @return bool
   *   Whether successful.
   */
  protected function doSendWithEvents(InternalEmailInterface $email): bool {
    // Use reflection to access the protected doSend method's logic.
    // We need to replicate what the original Mailer does, but with our
    // transport creation.
    //
    // The cleanest approach is to use the inner mailer for everything
    // except the final transport creation and send.
    // Let the inner mailer handle all the processing phases (build,
    // pre-render, render, post-render) by calling send(), but we need
    // to intercept the actual SMTP sending.
    //
    // Unfortunately, the inner mailer's send() method does the SMTP send
    // internally. We need a different approach.
    //
    // Best approach: Use reflection to call the processing parts, then
    // do the actual send ourselves with the dispatcher-enabled transport.
    return $this->sendWithReflection($email);
  }

  /**
   * Sends email using reflection to access inner mailer's processing.
   */
  protected function sendWithReflection(InternalEmailInterface $email): bool {
    $mailer = $this->innerMailer;
    $reflection = new \ReflectionClass($mailer);

    // Get the renderer to execute in render context.
    $rendererProp = $reflection->getProperty('renderer');
    $rendererProp->setAccessible(TRUE);
    $renderer = $rendererProp->getValue($mailer);

    // Execute in render context like the original.
    return $renderer->executeInRenderContext(
      new RenderContext(),
      function () use ($email, $mailer, $reflection) {
        try {
          return $this->doSendInternal($email, $mailer, $reflection);
        }
        catch (SkipMailException $e) {
          $accountProp = $reflection->getProperty('account');
          $accountProp->setAccessible(TRUE);
          $account = $accountProp->getValue($mailer);

          if ($account->hasPermission('administer mailer')) {
            $this->messenger->addError($this->t('Email sending skipped: %message.', [
              '%message' => $e->getMessage(),
            ]));
          }
          return FALSE;
        }
      }
    );
  }

  /**
   * Internal send implementation with event dispatcher injection.
   */
  protected function doSendInternal(InternalEmailInterface $email, MailerInterface $mailer, \ReflectionClass $reflection): bool {
    // Get required services from inner mailer via reflection.
    $themeManager = $this->getProperty($mailer, $reflection, 'themeManager');
    $languageManager = $this->getProperty($mailer, $reflection, 'languageManager');
    $account = $this->getProperty($mailer, $reflection, 'account');
    $accountSwitcher = $this->getProperty($mailer, $reflection, 'accountSwitcher');
    $loggerFactory = $this->getProperty($mailer, $reflection, 'loggerFactory');

    // Save active theme.
    $active_theme_name = $themeManager->getActiveTheme()->getName();

    // Process the build phase.
    $email->process();

    // Handle language and account switching (simplified version).
    $current_langcode = $languageManager->getCurrentLanguage()->getId();

    if ($email->getParam('__disable_customize__')) {
      $langcode = $current_langcode;
      $switch_account = $account;
    }
    else {
      $mailer->changeTheme($email->getTheme());

      $langcodes = $accounts = [];
      foreach ($email->getTo() as $to) {
        if ($loop_langcode = $to->getLangcode()) {
          $langcodes[$loop_langcode] = $loop_langcode;
        }
        if ($loop_account = $to->getAccount()) {
          $accounts[$loop_account->id()] = $loop_account;
        }
      }
      $langcode = (count($langcodes) == 1) ? reset($langcodes) : $languageManager->getDefaultLanguage()->getId();
      $switch_account = (count($accounts) == 1) ? reset($accounts) : User::getAnonymousUser();
    }

    $email->customize($langcode, $switch_account);

    $must_switch_account = $switch_account->id() != $account->id();
    if ($must_switch_account) {
      $accountSwitcher->switchTo($switch_account);
    }

    $must_switch_language = $langcode !== $current_langcode;
    if ($must_switch_language) {
      $this->changeActiveLanguage($mailer, $reflection, $langcode);
    }

    try {
      // Process pre-render, render, post-render phases.
      $email->process();
      $email->render();
      $email->process();
    }
    finally {
      if ($must_switch_account) {
        $accountSwitcher->switchBack();
      }
      if ($must_switch_language) {
        $this->changeActiveLanguage($mailer, $reflection, $current_langcode);
      }
      $mailer->changeTheme($active_theme_name);
    }

    // Send with event dispatcher injected into transport.
    try {
      $symfony_email = $email->getSymfonyEmail();
      $transport_dsn = $email->getTransportDsn();

      if (empty($transport_dsn)) {
        throw new MissingTransportException();
      }

      // Key fix: Use Transport::fromDsn() with event dispatcher.
      $transport = Transport::fromDsn($transport_dsn, $this->eventDispatcher);
      $symfonyMailer = new SymfonyMailer($transport);
      $symfonyMailer->send($symfony_email);
      $result = TRUE;
    }
    catch (\Exception $e) {
      if ($e instanceof MissingTransportException) {
        $message = $this->t('Missing email transport: please <a href=":url">configure a default</a>.', [
          ':url' => Url::fromRoute('entity.mailer_transport.collection')->toString(),
        ]);
      }
      else {
        $message = $e->getMessage();
      }
      // Method setError() expects a string, not TranslatableMarkup.
      $email->setError((string) $message);

      $loggerFactory->get('symfony_mailer')->error('Error sending email: %message', ['%message' => $message]);

      if (!$account->hasPermission('administer mailer')) {
        $message = $this->t('Unable to send email. Contact the site administrator if the problem persists.');
      }

      $this->messenger->addError($message);
      $result = FALSE;
    }

    // Process post-send phase.
    $email->process();

    return $result;
  }

  /**
   * Gets a property value from the inner mailer via reflection.
   */
  protected function getProperty(object $object, \ReflectionClass $reflection, string $property): mixed {
    $prop = $reflection->getProperty($property);
    $prop->setAccessible(TRUE);
    return $prop->getValue($object);
  }

  /**
   * Changes the active language (replicated from original Mailer).
   */
  protected function changeActiveLanguage(MailerInterface $mailer, \ReflectionClass $reflection, string $langcode): void {
    $languageManager = $this->getProperty($mailer, $reflection, 'languageManager');
    $languageDefault = $this->getProperty($mailer, $reflection, 'languageDefault');

    if (!$languageManager->isMultilingual()) {
      return;
    }

    $language = $languageManager->getLanguage($langcode);
    if (!$language) {
      return;
    }

    $languageDefault->set($language);
    $languageManager->setConfigOverrideLanguage($language);
    $languageManager->reset();

    $string_translation = $this->getStringTranslation();
    if ($string_translation instanceof TranslationManager) {
      $string_translation->setDefaultLangcode($language->getId());
      $string_translation->reset();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function changeTheme(string $theme_name): string {
    return $this->innerMailer->changeTheme($theme_name);
  }

}
