<?php

declare(strict_types=1);

namespace Drupal\trace_mail_log;

use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\DependencyInjection\ServiceProviderBase;
use Drupal\trace_mail_log\EventSubscriber\MailerEventSubscriber;
use Drupal\trace_mail_log\EventSubscriber\QueueEventSubscriber;
use Drupal\trace_mail_log\Service\WebformEmailStatusService;
use Symfony\Component\DependencyInjection\Reference;

/**
 * Service provider for trace_mail_log module.
 *
 * Registers mail logging decorators based on which mail system is installed.
 * - For Drupal core mail: decorates plugin.manager.mail (always)
 * - For Symfony Mailer: decorates symfony_mailer service (when available)
 * - For Webform: registers email status service (when available)
 *
 * Uses alter() instead of register() to ensure all module services are
 * available for inspection before conditionally registering decorators.
 */
class TraceMailLogServiceProvider extends ServiceProviderBase {

  /**
   * {@inheritdoc}
   */
  public function alter(ContainerBuilder $container): void {
    // Check if symfony_mailer service exists (module is installed/enabled).
    if ($container->has('symfony_mailer')) {
      // Register Symfony Mailer decorator for full transcript support.
      $container
        ->register('trace_mail_log.mailer_decorator', MailerDecorator::class)
        ->setDecoratedService('symfony_mailer')
        ->addArgument(new Reference('trace_mail_log.mailer_decorator.inner'))
        ->addArgument(new Reference('event_dispatcher'))
        ->addArgument(new Reference('messenger'))
        ->addArgument(new Reference('string_translation'));

      // Register the Symfony Mailer event subscriber.
      $container
        ->register('trace_mail_log.mailer_event_subscriber', MailerEventSubscriber::class)
        ->addArgument(new Reference('trace_mail_log.mail_log_service'))
        ->addArgument(new Reference('config.factory'))
        ->addTag('event_subscriber');

      // Register queue event subscriber (for symfony_mailer_queue).
      $container
        ->register('trace_mail_log.queue_event_subscriber', QueueEventSubscriber::class)
        ->addArgument(new Reference('trace_mail_log.mail_log_service'))
        ->addArgument(new Reference('config.factory'))
        ->addTag('event_subscriber');
    }
    else {
      // No symfony_mailer - use the MailManager decorator for core/mailsystem.
      $container->register('trace_mail_log.mail_manager_decorator', MailManagerDecorator::class)
        ->setDecoratedService('plugin.manager.mail')
        ->addArgument(new Reference('trace_mail_log.mail_manager_decorator.inner'))
        ->addArgument(new Reference('trace_mail_log.mail_log_service'));
    }

    // Register webform email status service if webform is installed.
    // Check for webform's message manager service as indicator.
    if ($container->has('webform.message_manager')) {
      $container->register('trace_mail_log.webform_email_status', WebformEmailStatusService::class)
        ->addArgument(new Reference('database'))
        ->addArgument(new Reference('cache.static'));
    }
  }

}
