<?php

declare(strict_types=1);

namespace Drupal\trace_mail_log;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Routing\RouteProviderInterface;
use Drupal\Core\Url;
use Drupal\trace_mail_log\Service\WebformEmailStatusService;
use Drupal\webform\WebformSubmissionInterface;
use Drupal\webform\WebformSubmissionListBuilder;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Extends WebformSubmissionListBuilder to add email status column.
 */
class WebformSubmissionEmailStatusListBuilder extends WebformSubmissionListBuilder {

  /**
   * The email status service.
   */
  protected WebformEmailStatusService $emailStatusService;

  /**
   * The route provider service.
   */
  protected RouteProviderInterface $routeProvider;

  /**
   * Whether the current webform has email handlers.
   */
  protected ?bool $hasEmailHandlers = NULL;

  /**
   * Cached result of route existence check.
   */
  protected ?bool $traceMailLogRouteExists = NULL;

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type) {
    /** @var \Drupal\trace_mail_log\WebformSubmissionEmailStatusListBuilder $instance */
    $instance = parent::createInstance($container, $entity_type);
    $instance->emailStatusService = $container->get('trace_mail_log.webform_email_status');
    $instance->routeProvider = $container->get('router.route_provider');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function initialize() {
    parent::initialize();

    // Add email status column if this webform has email handlers.
    if ($this->webform && $this->hasEmailHandlers() && is_array($this->columns)) {
      $this->addEmailStatusColumn();
    }
  }

  /**
   * Checks if the current webform has any enabled email handlers.
   *
   * @return bool
   *   TRUE if the webform has enabled email handlers.
   */
  protected function hasEmailHandlers(): bool {
    if ($this->hasEmailHandlers !== NULL) {
      return $this->hasEmailHandlers;
    }

    $this->hasEmailHandlers = FALSE;

    if (!$this->webform) {
      return $this->hasEmailHandlers;
    }

    $handlers = $this->webform->getHandlers();
    foreach ($handlers as $handler) {
      if ($handler->getPluginId() === 'email' && $handler->isEnabled()) {
        $this->hasEmailHandlers = TRUE;
        break;
      }
    }

    return $this->hasEmailHandlers;
  }

  /**
   * Adds the email status column to the columns array.
   */
  protected function addEmailStatusColumn(): void {
    // Insert email_status column after 'created'.
    $new_columns = [];
    foreach ($this->columns as $key => $column) {
      $new_columns[$key] = $column;
      if ($key === 'created') {
        $new_columns['email_status'] = [
          'name' => 'email_status',
          'title' => $this->t('Email'),
          'sort' => FALSE,
          'format' => 'value',
        ];
      }
    }

    // If 'created' column wasn't found, add at the end.
    if (!isset($new_columns['email_status'])) {
      $new_columns['email_status'] = [
        'name' => 'email_status',
        'title' => $this->t('Email'),
        'sort' => FALSE,
        'format' => 'value',
      ];
    }

    $this->columns = $new_columns;
  }

  /**
   * {@inheritdoc}
   */
  public function buildRowColumn(array $column, EntityInterface $entity) {
    $name = $column['name'];

    if ($name === 'email_status') {
      return $this->buildEmailStatusColumn($entity);
    }

    return parent::buildRowColumn($column, $entity);
  }

  /**
   * Builds the email status column content.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The webform submission entity.
   *
   * @return array{data: array}
   *   A render array for the email status.
   */
  protected function buildEmailStatusColumn(EntityInterface $entity): array {
    if (!$entity instanceof WebformSubmissionInterface) {
      return ['data' => ['#markup' => '']];
    }

    $status_info = $this->emailStatusService->getEmailStatus($entity);
    $status = $status_info['status'];

    // Try to get log_id from details for linking.
    $log_id = $this->getLogIdFromDetails($status_info['details'] ?? []);

    // Only create links if the route exists.
    $can_link = $log_id && $this->traceMailLogRouteExists();

    switch ($status) {
      case WebformEmailStatusService::STATUS_SUCCESS:
        return $this->buildStatusCell(
          $this->t('Sent'),
          ['email-status', 'email-status--success'],
          $this->t('Email was sent successfully'),
          $can_link ? $log_id : NULL
        );

      case WebformEmailStatusService::STATUS_FAILED:
        $message = $this->getResponseMessage($status_info['details'] ?? []);
        return $this->buildStatusCell(
          $this->t('Failed'),
          ['email-status', 'email-status--failed'],
          $message !== '' ? $message : $this->t('Email sending failed'),
          $can_link ? $log_id : NULL
        );

      case WebformEmailStatusService::STATUS_PENDING:
        return $this->buildStatusCell(
          $this->t('Pending'),
          ['email-status', 'email-status--pending'],
          $this->t('Email is in queue, waiting to be sent'),
          $can_link ? $log_id : NULL
        );

      case WebformEmailStatusService::STATUS_NO_HANDLER:
        return $this->buildStatusCell(
          '-',
          ['email-status', 'email-status--no-handler'],
          $this->t('No email handler configured'),
          NULL
        );

      case WebformEmailStatusService::STATUS_UNKNOWN:
      default:
        return $this->buildStatusCell(
          '?',
          ['email-status', 'email-status--unknown'],
          $this->t('Email status unknown'),
          NULL
        );
    }
  }

  /**
   * Checks if the trace_mail_log.detail route exists.
   *
   * @return bool
   *   TRUE if the route exists.
   */
  protected function traceMailLogRouteExists(): bool {
    if ($this->traceMailLogRouteExists !== NULL) {
      return $this->traceMailLogRouteExists;
    }

    try {
      $this->routeProvider->getRouteByName('trace_mail_log.detail');
      $this->traceMailLogRouteExists = TRUE;
    }
    catch (\Exception $e) {
      $this->traceMailLogRouteExists = FALSE;
    }

    return $this->traceMailLogRouteExists;
  }

  /**
   * Builds a status cell with optional link to trace_mail_log.
   *
   * @param string|\Drupal\Core\StringTranslation\TranslatableMarkup $text
   *   The status text.
   * @param array $classes
   *   CSS classes.
   * @param string|\Drupal\Core\StringTranslation\TranslatableMarkup $title
   *   The title attribute.
   * @param int|null $log_id
   *   The trace_mail_log entry ID, if available.
   *
   * @return array{data: array}
   *   A render array.
   */
  protected function buildStatusCell($text, array $classes, $title, ?int $log_id): array {
    if ($log_id) {
      return [
        'data' => [
          '#type' => 'link',
          '#title' => $text,
          '#url' => Url::fromRoute('trace_mail_log.detail', ['id' => $log_id]),
          '#attributes' => [
            'class' => $classes,
            'title' => $title,
            'target' => '_blank',
          ],
          '#attached' => [
            'library' => ['trace_mail_log/webform_email_status'],
          ],
        ],
      ];
    }

    return [
      'data' => [
        '#type' => 'html_tag',
        '#tag' => 'span',
        '#value' => $text,
        '#attributes' => [
          'class' => $classes,
          'title' => $title,
        ],
        '#attached' => [
          'library' => ['trace_mail_log/webform_email_status'],
        ],
      ],
    ];
  }

  /**
   * Gets the log ID from status details.
   *
   * @param array $details
   *   The status details array.
   *
   * @return int|null
   *   The log ID if found.
   */
  protected function getLogIdFromDetails(array $details): ?int {
    // Check for direct log_id in details.
    if (isset($details['log_id'])) {
      return (int) $details['log_id'];
    }

    // Check in handler-specific details (when multiple handlers).
    foreach ($details as $handler_details) {
      if (is_array($handler_details) && isset($handler_details['log_id'])) {
        return (int) $handler_details['log_id'];
      }
    }

    return NULL;
  }

  /**
   * Gets the response message from status details.
   *
   * @param array $details
   *   The status details array.
   *
   * @return string
   *   The response message if found.
   */
  protected function getResponseMessage(array $details): string {
    // Check for direct response_message.
    if (isset($details['response_message'])) {
      return (string) $details['response_message'];
    }

    // Check in handler-specific details.
    foreach ($details as $handler_details) {
      if (is_array($handler_details) && isset($handler_details['response_message'])) {
        return (string) $handler_details['response_message'];
      }
    }

    return '';
  }

}
