<?php

declare(strict_types=1);

namespace Drupal\Tests\trace_mail_log\Unit;

use Drupal\Tests\UnitTestCase;
use Drupal\trace_mail_log\Service\MailLogService;

/**
 * Tests transport parsing methods in MailLogService.
 *
 * @coversDefaultClass \Drupal\trace_mail_log\Service\MailLogService
 * @group trace_mail_log
 */
class TransportParserTest extends UnitTestCase {

  /**
   * The mail log service mock for testing protected methods.
   *
   * @var \Drupal\trace_mail_log\Service\MailLogService
   */
  protected MailLogService $service;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create a partial mock to test public methods that use parsing.
    // We need to mock all constructor dependencies.
    $database = $this->createMock('Drupal\Core\Database\Connection');
    $file_system = $this->createMock('Drupal\Core\File\FileSystemInterface');
    $logger_factory = $this->createMock('Drupal\Core\Logger\LoggerChannelFactoryInterface');
    $logger_factory->method('get')->willReturn($this->createMock('Psr\Log\LoggerInterface'));
    $config_factory = $this->createMock('Drupal\Core\Config\ConfigFactoryInterface');
    $time = $this->createMock('Drupal\Component\Datetime\TimeInterface');
    $uuid = $this->createMock('Drupal\Component\Uuid\UuidInterface');

    $this->service = new MailLogService(
      $database,
      $file_system,
      $logger_factory,
      $config_factory,
      $time,
      $uuid
    );
  }

  /**
   * Tests parseTransportResponse with successful response.
   *
   * @covers ::parseTransportResponse
   */
  public function testParseTransportResponseSuccess(): void {
    $transcript = "220 mail.example.com ESMTP\r\n250 OK\r\n250 2.0.0 Ok: queued as ABC123";

    $result = $this->service->parseTransportResponse($transcript);

    $this->assertEquals(250, $result['code']);
    $this->assertEquals('2.0.0 Ok: queued as ABC123', $result['message']);
  }

  /**
   * Tests parseTransportResponse with error response.
   *
   * @covers ::parseTransportResponse
   */
  public function testParseTransportResponseError(): void {
    $transcript = "220 mail.example.com ESMTP\r\n550 Mailbox not found";

    $result = $this->service->parseTransportResponse($transcript);

    $this->assertEquals(550, $result['code']);
    $this->assertEquals('Mailbox not found', $result['message']);
  }

  /**
   * Tests parseTransportResponse with multi-line response.
   *
   * @covers ::parseTransportResponse
   */
  public function testParseTransportResponseMultiLine(): void {
    $transcript = <<<TRANSCRIPT
220 mail.example.com ESMTP Postfix
250-mail.example.com Hello client.example.com
250-SIZE 52428800
250-8BITMIME
250 DSN
354 End data with <CR><LF>.<CR><LF>
250 2.0.0 Ok: queued as 1234567890
TRANSCRIPT;

    $result = $this->service->parseTransportResponse($transcript);

    $this->assertEquals(250, $result['code']);
    $this->assertStringContainsString('queued', $result['message']);
  }

  /**
   * Tests parseTransportResponse with empty transcript.
   *
   * @covers ::parseTransportResponse
   */
  public function testParseTransportResponseEmpty(): void {
    $result = $this->service->parseTransportResponse('');

    $this->assertNull($result['code']);
    $this->assertNull($result['message']);
  }

  /**
   * Tests parseTransportResponse with temporary error.
   *
   * @covers ::parseTransportResponse
   */
  public function testParseTransportResponseTemporaryError(): void {
    $transcript = "220 mail.example.com ESMTP\r\n421 Service temporarily unavailable";

    $result = $this->service->parseTransportResponse($transcript);

    $this->assertEquals(421, $result['code']);
    $this->assertEquals('Service temporarily unavailable', $result['message']);
  }

  /**
   * Tests parseTransportResponse with authentication failure.
   *
   * @covers ::parseTransportResponse
   */
  public function testParseTransportResponseAuthFailure(): void {
    $transcript = "220 mail.example.com ESMTP\r\n535 5.7.8 Authentication credentials invalid";

    $result = $this->service->parseTransportResponse($transcript);

    $this->assertEquals(535, $result['code']);
    $this->assertStringContainsString('Authentication', $result['message']);
  }

  /**
   * Tests parseTransportResponse with Symfony Mailer debug format (timestamps).
   *
   * @covers ::parseTransportResponse
   */
  public function testParseTransportResponseSymfonyDebugFormat(): void {
    $transcript = <<<TRANSCRIPT
[2025-12-12T08:20:11.606312+01:00] < 220 mail-server ESMTP Service ready
[2025-12-12T08:20:11.606341+01:00] > EHLO [127.0.0.1]
[2025-12-12T08:20:11.606390+01:00] < 250-mail-server greets [127.0.0.1]
[2025-12-12T08:20:11.606393+01:00] < 250-SIZE 0
[2025-12-12T08:20:11.606394+01:00] < 250 SMTPUTF8
[2025-12-12T08:20:11.606424+01:00] > MAIL FROM:<sender@example.com>
[2025-12-12T08:20:11.606475+01:00] < 250 2.1.0 Ok
[2025-12-12T08:20:11.606479+01:00] > RCPT TO:<recipient@example.com>
[2025-12-12T08:20:11.606504+01:00] < 250 2.1.5 Ok
[2025-12-12T08:20:11.606506+01:00] > DATA
[2025-12-12T08:20:11.606518+01:00] < 354 Start mail input; end with <CR><LF>.<CR><LF>
[2025-12-12T08:20:11.665911+01:00] > .
[2025-12-12T08:20:11.671480+01:00] < 250 2.0.0 Ok: queued as ABC123XYZ
TRANSCRIPT;

    $result = $this->service->parseTransportResponse($transcript);

    $this->assertEquals(250, $result['code']);
    $this->assertEquals('2.0.0 Ok: queued as ABC123XYZ', $result['message']);
  }

  /**
   * Tests parseTransportResponse with Symfony debug format and error.
   *
   * @covers ::parseTransportResponse
   */
  public function testParseTransportResponseSymfonyDebugError(): void {
    $transcript = <<<TRANSCRIPT
[2025-12-12T08:20:11.606312+01:00] < 220 mail-server ESMTP Service ready
[2025-12-12T08:20:11.606424+01:00] > MAIL FROM:<sender@example.com>
[2025-12-12T08:20:11.606475+01:00] < 550 5.1.1 User unknown
TRANSCRIPT;

    $result = $this->service->parseTransportResponse($transcript);

    $this->assertEquals(550, $result['code']);
    $this->assertStringContainsString('User unknown', $result['message']);
  }

  /**
   * Data provider for transport type parsing tests.
   *
   * @return array
   *   Test cases with DSN and expected transport type.
   */
  public static function transportTypeProvider(): array {
    return [
      'smtp' => ['smtp://localhost:25', 'smtp'],
      'smtps' => ['smtps://mail.gmail.com:465', 'smtp'],
      'smtp with auth' => ['smtp://user:pass@mail.example.com:587', 'smtp'],
      'sendmail' => ['sendmail://default', 'sendmail'],
      'sendmail with path' => ['sendmail:///usr/sbin/sendmail', 'sendmail'],
      // Symfony Mailer's SendmailTransport uses "smtp://sendmail" as its
      // string representation.
      'sendmail via symfony' => ['smtp://sendmail', 'sendmail'],
      'native' => ['native://default', 'native'],
      'null' => ['null://null', 'null'],
      'unknown scheme' => ['custom://server', 'unknown'],
      'empty string' => ['', 'unknown'],
      'no scheme' => ['localhost:25', 'unknown'],
    ];
  }

  /**
   * Tests transport type parsing from DSN.
   *
   * @param string $dsn
   *   The DSN string.
   * @param string $expected
   *   The expected transport type.
   *
   * @covers ::parseTransportType
   * @dataProvider transportTypeProvider
   */
  public function testParseTransportType(string $dsn, string $expected): void {
    // Use reflection to access protected method.
    $reflection = new \ReflectionClass($this->service);
    $method = $reflection->getMethod('parseTransportType');
    $method->setAccessible(TRUE);

    $result = $method->invoke($this->service, $dsn);

    $this->assertEquals($expected, $result);
  }

}
