<?php

declare(strict_types=1);

namespace Drupal\trace_mail_log\EventSubscriber;

use Drupal\trace_mail_log\Service\MailLogService;
use Drupal\Core\Config\ConfigFactoryInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\Mime\Email;

/**
 * Subscribes to symfony_mailer_queue events for logging.
 *
 * Note: This subscriber listens for queue-specific events when
 * the symfony_mailer_queue module is enabled.
 */
class QueueEventSubscriber implements EventSubscriberInterface {

  /**
   * The mail log service.
   */
  protected MailLogService $mailLogService;

  /**
   * The config factory.
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Constructs a QueueEventSubscriber.
   */
  public function __construct(
    MailLogService $mail_log_service,
    ConfigFactoryInterface $config_factory,
  ) {
    $this->mailLogService = $mail_log_service;
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    $events = [];

    // Only subscribe if symfony_mailer_queue is available.
    if (class_exists('Drupal\symfony_mailer_queue\Event\EmailSendFailureEvent')) {
      $events['Drupal\symfony_mailer_queue\Event\EmailSendFailureEvent'] = ['onQueueFailure', 0];
    }

    if (class_exists('Drupal\symfony_mailer_queue\Event\EmailSendRequeueEvent')) {
      $events['Drupal\symfony_mailer_queue\Event\EmailSendRequeueEvent'] = ['onQueueRequeue', 0];
    }

    return $events;
  }

  /**
   * Handles queue failure event.
   *
   * @param object $event
   *   The queue failure event.
   */
  public function onQueueFailure(object $event): void {
    if (!$this->mailLogService->isEnabled()) {
      return;
    }

    // Get the email from the event.
    if (!method_exists($event, 'getEmail')) {
      return;
    }

    $email = $event->getEmail();
    if (!$email instanceof Email) {
      return;
    }

    $uuid = $this->getUuid($email);
    if (!$uuid) {
      return;
    }

    $data = $this->mailLogService->extractEmailData($email);
    $data['mail_key'] = $this->getMailKey($email);

    // Get error if available.
    if (method_exists($event, 'getError')) {
      $data['error'] = $event->getError()->getMessage();
    }

    $this->mailLogService->log($uuid, 'failed', 'failed', $data);
  }

  /**
   * Handles queue requeue event.
   *
   * @param object $event
   *   The requeue event.
   */
  public function onQueueRequeue(object $event): void {
    if (!$this->mailLogService->isEnabled()) {
      return;
    }

    if (!method_exists($event, 'getEmail')) {
      return;
    }

    $email = $event->getEmail();
    if (!$email instanceof Email) {
      return;
    }

    $uuid = $this->getUuid($email);
    if (!$uuid) {
      return;
    }

    $data = $this->mailLogService->extractEmailData($email);
    $data['mail_key'] = $this->getMailKey($email);

    // Get attempt number if available.
    if (method_exists($event, 'getAttempt')) {
      $data['attempt'] = $event->getAttempt();
    }

    $this->mailLogService->log($uuid, 'requeued', 'pending', $data);
  }

  /**
   * Gets the tracking UUID from a message.
   */
  protected function getUuid(Email $message): ?string {
    $headers = $message->getHeaders();

    if ($headers->has(MailerEventSubscriber::UUID_HEADER)) {
      return $headers->get(MailerEventSubscriber::UUID_HEADER)->getBodyAsString();
    }

    return NULL;
  }

  /**
   * Gets the Drupal mail key from message headers.
   */
  protected function getMailKey(Email $message): ?string {
    $headers = $message->getHeaders();

    if ($headers->has(MailerEventSubscriber::MAIL_KEY_HEADER)) {
      return $headers->get(MailerEventSubscriber::MAIL_KEY_HEADER)->getBodyAsString();
    }

    return NULL;
  }

}
