<?php

declare(strict_types=1);

namespace Drupal\track_usage\Drush\Commands;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\track_usage\Entity\TrackConfigInterface;
use Drupal\track_usage\Model\BulkUpdateMethod;
use Drupal\track_usage\UpdaterInterface;
use Drush\Attributes as CLI;
use Drush\Boot\DrupalBootLevels;
use Drush\Commands\AutowireTrait;
use Drush\Commands\DrushCommands;

/**
 * Drush commands.
 */
class TrackUsageCommands extends DrushCommands {

  use AutowireTrait;

  public function __construct(
    protected readonly UpdaterInterface $updater,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct();
  }

  /**
   * Update usages.
   */
  #[CLI\Command(name: 'track_usage:update')]
  #[CLI\Argument(name: 'config', description: 'The track usage configuration to be used')]
  #[CLI\Option(name: 'method', description: 'Update method: batch, queue or instant (not recommended). Default: batch')]
  #[CLI\Usage(name: 'drush track_usage:update default', description: 'Recreate the "default" usage records using batch process')]
  #[CLI\Usage(name: 'drush track_usage:update file_usages --method=queue', description: 'Pass the "file_update" records update to a queue which runs on cron')]
  #[CLI\Usage(name: 'drush track_usage:update default --method=instant', description: '[not recommended] Update "default" records instantly')]
  #[CLI\Bootstrap(level: DrupalBootLevels::FULL)]
  public function update(string $config, array $options = ['method' => 'batch']): void {
    $methodEnum = BulkUpdateMethod::tryFrom($options['method']);
    if (!$methodEnum) {
      throw new \InvalidArgumentException("Invalid method: $options[method]");
    }

    $configuration = $this->entityTypeManager->getStorage('track_usage_config')->load($config);
    if (!$configuration instanceof TrackConfigInterface) {
      throw new \InvalidArgumentException("Invalid track usage configuration: $config");
    }

    $this->updater->update($methodEnum, $configuration);
    if ($methodEnum === BulkUpdateMethod::Batch) {
      drush_backend_batch_process();
    }
  }

}
