<?php

declare(strict_types=1);

namespace Drupal\track_usage\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\Entity\ConfigEntityStorageInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\track_usage\Model\BulkUpdateMethod;
use Drupal\track_usage\UpdaterInterface;

/**
 * Settings form.
 */
class TrackUsageSettingsForm extends ConfigFormBase {

  use AutowireTrait;

  public function __construct(
    ConfigFactoryInterface $configFactory,
    TypedConfigManagerInterface $typedConfigManager,
    protected UpdaterInterface $updater,
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($configFactory, $typedConfigManager);
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['bulk_update'] = [
      '#type' => 'radios',
      '#title' => $this->t('Bulk update mode'),
      '#options' => BulkUpdateMethod::asOptions(),
      '#config_target' => 'track_usage.settings:bulk_update',
    ];

    $form['actions']['bulk_update'] = [
      '#type' => 'submit',
      '#value' => $this->t('Bulk update now'),
      '#submit' => ['::submitForm', '::bulkUpdateSubmit'],
      '#weight' => 10,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Provides a submit callback that updates all usages.
   *
   * @param array $form
   *   The form API render array.
   * @param \Drupal\Core\Form\FormStateInterface $formState
   *   The form API state object.
   */
  public function bulkUpdateSubmit(array &$form, FormStateInterface $formState): void {
    $method = $formState->getValue('bulk_update');

    $storage = $this->entityTypeManager->getStorage('track_usage_config');
    assert($storage instanceof ConfigEntityStorageInterface);
    $ids = $storage->getQuery()->condition('status', TRUE)->execute();
    foreach ($storage->loadMultiple($ids) as $config) {
      $this->updater->update(BulkUpdateMethod::from($method), $config);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'track_usage_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['track_usage.settings'];
  }

}
