<?php

declare(strict_types=1);

namespace Drupal\track_usage\Model;

use Drupal\Component\Render\MarkupInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\track_usage\Attribute\EntityRoleMeta as Meta;

/**
 * The entity role in the track usage process.
 */
enum EntityRole: string {

  #[Meta(
    title: new TranslatableMarkup('Source entity types'),
    description: new TranslatableMarkup('Source entities are the ones for witch you track usages for target entities (e.g., node).'),
  )]
  case Source = 'source';

  #[Meta(
    title: new TranslatableMarkup('Traversable entity types'),
    description: new TranslatableMarkup('Traversable entities are in the middle, between source and target entities (e.g., media, paragraph).'),
    isTraversable: TRUE,
  )]
  case Traversable = 'traversable';

  #[Meta(
    title: new TranslatableMarkup('Target entity types'),
    description: new TranslatableMarkup('Target entities are entities whose usage is tracked.'),
    limitToFieldable: FALSE,
  )]
  case Target = 'target';

  /**
   * Checks if the role is traversable.
   *
   * @return bool
   *   If the role is traversable.
   */
  public function isTraversable(): bool {
    $reflection = new \ReflectionEnumBackedCase(static::class, $this->name);
    $attributes = $reflection->getAttributes(Meta::class);
    return $attributes[0]->getArguments()['isTraversable'] ?? FALSE;
  }

  /**
   * Checks if the role is designated only for fieldable entity types.
   *
   * @return bool
   *   If the entity role is designated only for fieldable entity types.
   */
  public function onlyFieldableEntityTypes(): bool {
    $reflection = new \ReflectionEnumBackedCase(static::class, $this->name);
    $attributes = $reflection->getAttributes(Meta::class);
    return $attributes[0]->getArguments()['limitToFieldable'] ?? TRUE;
  }

  /**
   * Returns the translatable title of the entity role.
   *
   * @return \Drupal\Component\Render\MarkupInterface
   *   The translatable title of the entity role.
   */
  public function getTitle(): MarkupInterface {
    $reflection = new \ReflectionEnumBackedCase(static::class, $this->name);
    $attributes = $reflection->getAttributes(Meta::class);
    return $attributes[0]->getArguments()['title'];
  }

  /**
   * Returns the translatable description of the entity role.
   *
   * @return \Drupal\Component\Render\MarkupInterface
   *   The translatable description of the entity role.
   */
  public function getDescription(): MarkupInterface {
    $reflection = new \ReflectionEnumBackedCase($this, $this->name);
    $attributes = $reflection->getAttributes(Meta::class);
    return $attributes[0]->getArguments()['description'];
  }

}
