<?php

declare(strict_types=1);

namespace Drupal\track_usage\Model;

use Drupal\Core\Entity\EntityInterface;

/**
 * Collection of entity usages.
 */
class UsageCollection implements \IteratorAggregate, \Countable {

  /**
   * Internal storage.
   *
   * @var list<\Drupal\track_usage\Model\Usage>
   */
  private array $storage = [];

  /**
   * Adds a usage record.
   *
   * @return $this
   */
  public function add(EntityInterface $targetEntity, array $path): self {
    $paths = $this->storage[$targetEntity->id()]->paths ?? [];
    if (!in_array($path, $paths, TRUE)) {
      $paths[] = $path;
    }
    $this->storage[$targetEntity->id()] = new Usage($targetEntity, $paths);
    return $this;
  }

  /**
   * Returns all usages.
   *
   * @return list<\Drupal\track_usage\Model\Usage>
   *   List of usages.
   */
  public function all(): array {
    return $this->getIterator()->getArrayCopy();
  }

  /**
   * Merges other collection.
   *
   * @param \Drupal\track_usage\Model\UsageCollection $other
   *   A collection to be merged in.
   *
   * @return $this
   */
  public function merge(UsageCollection $other): self {
    foreach ($other->getIterator() as $usage) {
      $this->add($usage->getEntity(), $usage->getPath());
    }
    return $this;
  }

  /**
   * Returns an array representation of the collection.
   *
   * @return list<array{string|int, list<list<string>>}>
   *   An array representation of the collection.
   */
  public function toArray(): array {
    return array_map(
      fn(Usage $usage): array => $usage->toArray(),
      array_values($this->all()),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getIterator(): \ArrayIterator {
    return new \ArrayIterator($this->storage);
  }

  /**
   * {@inheritdoc}
   */
  public function count(): int {
    return $this->getIterator()->count();
  }

}
