<?php

declare(strict_types=1);

namespace Drupal\track_usage\Plugin\TrackUsage\Track;

use Drupal\Component\Utility\Html;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\FieldItemInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\filter\FilterFormatInterface;
use Drupal\track_usage\Attribute\Track;
use Drupal\track_usage\EntityGuesser;
use Drupal\track_usage\EntityGuesserInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Tracks usage of entities referenced from regular HTML links.
 */
#[Track(
  id: 'html_link',
  label: new TranslatableMarkup('HTML links'),
  fieldTypes: ['text', 'text_long', 'text_with_summary'],
)]
class HtmlLink extends TextFieldBase {

  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    EntityRepositoryInterface $entityRepository,
    EntityTypeManagerInterface $entityTypeManager,
    protected readonly ConfigFactoryInterface $configFactory,
    protected readonly EntityGuesserInterface $entityGuesser,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $entityRepository, $entityTypeManager);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity.repository'),
      $container->get('entity_type.manager'),
      $container->get('config.factory'),
      $container->get(EntityGuesser::class),
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function xpath(): string {
    return "//a[@href != '']";
  }

  /**
   * {@inheritdoc}
   */
  protected function getEntity(\DOMNode $node): ?EntityInterface {
    return $this->entityGuesser->guessFromUrl($node->getAttribute('href'));
  }

  /**
   * {@inheritdoc}
   */
  public function getItemTargetEntities(FieldItemInterface $item): iterable {
    $text = trim($item->value . '  ' . ($item->summary ?? ''));

    // If the text format convert URLs in text to real URLs we should too.
    if ($format_id = $item->format ?? $this->configFactory->get('filter.settings')->get('fallback_format')) {
      $format = $this->entityTypeManager->getStorage('filter_format')->load($format_id);
      if (
        $format instanceof FilterFormatInterface &&
        $format->filters()->has('filter_url') &&
        $format->filters('filter_url')->status
      ) {
        $filter = $format->filters()->get('filter_url');
        $text = $filter->process($text, $item->getLangcode())->getProcessedText();
      }
    }

    $document = Html::load($text);
    $xpath = new \DOMXPath($document);
    foreach ($xpath->query($this->xpath()) as $node) {
      if ($entity = $this->getEntity($node)) {
        yield $entity;
      }
    }
  }

}
