<?php

declare(strict_types=1);

namespace Drupal\track_usage\Plugin\TrackUsage\Track;

use Drupal\Component\Utility\Html;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Field\FieldItemInterface;
use Drupal\track_usage\Plugin\TrackUsage\TrackPluginBase;

/**
 * Base class for plugins tracking usage in entities embedded in text fields.
 */
abstract class TextFieldBase extends TrackPluginBase {

  /**
   * {@inheritdoc}
   */
  public function getItemTargetEntities(FieldItemInterface $item): iterable {
    $text = trim($item->value . '  ' . ($item->summary ?? ''));
    $document = Html::load($text);
    $xpath = new \DOMXPath($document);

    foreach ($xpath->query($this->xpath()) as $node) {
      if ($entity = $this->getEntity($node)) {
        yield $entity;
      }
    }
  }

  /**
   * Returns the XPath expression used to collect referenced entities from text.
   *
   * @return string
   *   The XPath expression used to collect referenced entities from text.
   */
  abstract protected function xpath(): string;

  /**
   * Returns a list of entities given an HTML element.
   *
   * @param \DOMNode $node
   *   The DOM node.
   *
   * @return \Drupal\Core\Entity\EntityInterface|null
   *   A list of entities.
   */
  protected function getEntity(\DOMNode $node): ?EntityInterface {
    $entityTypeId = $node->getAttribute('data-entity-type');
    $uuid = $node->getAttribute('data-entity-uuid');

    // Skip elements with empty data-entity-uuid/type attributes. Also, check
    // if the target entity still exists since text fields are not automatically
    // updated when an entity is removed.
    if ($entityTypeId && $uuid && $entity = $this->entityRepository->loadEntityByUuid($entityTypeId, $uuid)) {
      return $entity;
    }

    return NULL;
  }

}
