<?php

declare(strict_types=1);

namespace Drupal\track_usage\Plugin\TrackUsage;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Plugin\DefaultPluginManager;
use Drupal\track_usage\Attribute\Track;

/**
 * Plugin manager for track plugins.
 */
class TrackPluginManager extends DefaultPluginManager implements TrackPluginManagerInterface {

  /**
   * Static cache for applicable field types.
   *
   * @var list<non-empty-string>
   */
  protected array $applicableFieldTypes;

  public function __construct(
    \Traversable $namespaces,
    CacheBackendInterface $cacheBackend,
    ModuleHandlerInterface $moduleHandler,
    protected readonly ConfigFactoryInterface $configFactory,
  ) {
    parent::__construct(
      'Plugin/TrackUsage/Track',
      $namespaces,
      $moduleHandler,
      TrackPluginInterface::class,
      Track::class,
    );
    $this->alterInfo('file_track_usage_info');
    $this->setCacheBackend($cacheBackend, 'track_usage.track_plugins');
  }

  /**
   * {@inheritdoc}
   */
  public function getApplicablePlugins(string $fieldType): iterable {
    $definitions = $this->getDefinitions();
    $enabledPlugins = $this->configFactory->get('track_usage.settings')->get('track.plugins');
    $enabledPlugins = $enabledPlugins ?: array_keys($definitions);
    $definitions = array_intersect_key($definitions, array_flip($enabledPlugins));

    foreach ($definitions as $pluginId => $definition) {
      if (in_array($fieldType, $definition['fieldTypes'], TRUE)) {
        yield $this->createInstance($pluginId);
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getApplicableFieldTypes(): array {
    if (!isset($this->applicableFieldTypes)) {
      $definitions = $this->getDefinitions();
      $enabledPlugins = $this->configFactory->get('track_usage.settings')->get('track.plugins');
      $enabledPlugins = $enabledPlugins ?: array_keys($definitions);
      $definitions = array_intersect_key($definitions, array_flip($enabledPlugins));
      $this->applicableFieldTypes = array_reduce(
        $definitions,
        function (array $fieldTypes, array $definition): array {
          return [...$fieldTypes, ...array_diff($definition['fieldTypes'], $fieldTypes)];
        }, []);
    }
    return $this->applicableFieldTypes;
  }

}
