<?php

declare(strict_types=1);

namespace Drupal\track_usage\Trait;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\RevisionableInterface;
use Drupal\Core\Entity\TranslatableInterface;

/**
 * Reusable utility code.
 */
trait EntityUtilityTrait {

  /**
   * Returns the entity revision.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity revision or 0 if isn't supported.
   */
  protected function getRevisionId(EntityInterface $entity): int {
    return $entity instanceof RevisionableInterface ? (int) $entity->getRevisionId() : 0;
  }

  /**
   * Builds an "entity key".
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   Entity type, entity ID and revision separated by colon.
   */
  protected function getKey(EntityInterface $entity): string {
    return "{$entity->getEntityTypeId()}:{$entity->id()}:{$this->getRevisionId($entity)}";
  }

  /**
   * Checks whether an entity type is revisionable.
   *
   * @param \Drupal\Core\Entity\EntityInterface|string $entityOrEntityTypeId
   *   The entity or the entity type ID.
   *
   * @return bool
   *   Whether an entity type is revisionable.
   */
  protected static function isRevisionable(EntityInterface|string $entityOrEntityTypeId): bool {
    $entityTypeId = $entityOrEntityTypeId instanceof EntityInterface ? $entityOrEntityTypeId->getEntityTypeId() : $entityOrEntityTypeId;
    $entityType = \Drupal::entityTypeManager()->getDefinition($entityTypeId);
    return $entityType->entityClassImplements(RevisionableInterface::class) && $entityType->hasKey('revision');
  }

  /**
   * Loads a specific entity revision and translation.
   *
   * @param string $entityTypeId
   *   The entity type ID.
   * @param int|string $entityId
   *   The entity ID.
   * @param int|string $revisionId
   *   (optional) If passed, this revision will be loaded.
   * @param string|null $langcode
   *   (optional) If passed, this translation will be loaded.
   *
   * @return \Drupal\Core\Entity\EntityInterface|null
   *   The entity or NULL.
   */
  protected function loadEntity(string $entityTypeId, int|string $entityId, int|string $revisionId = 0, ?string $langcode = NULL): ?EntityInterface {
    assert(property_exists($this, 'entityTypeManager'));

    $storage = $this->entityTypeManager->getStorage($entityTypeId);

    if ($this->isRevisionable($entityTypeId) && $revisionId > 0) {
      $method = 'loadRevision';
      $id = $revisionId;
    }
    else {
      $method = 'load';
      $id = $entityId;
    }

    $entity = $storage->{$method}($id);

    if ($langcode && $entity instanceof TranslatableInterface) {
      try {
        $entity = $entity->getTranslation($langcode);
      }
      catch (\InvalidArgumentException) {
        // Translation doesn't exist.
        return NULL;
      }
    }

    return $entity;
  }

}
