<?php

declare(strict_types=1);

namespace Drupal\Tests\track_usage\Kernel;

use Drupal\Core\Entity\RevisionableStorageInterface;
use Drupal\file\Entity\File;
use Drupal\language\Entity\ConfigurableLanguage;
use Drupal\media\Entity\Media;
use Drupal\track_usage\Entity\TrackConfig;
use Drupal\track_usage\RecorderInterface;
use Drupal\track_usage\Trait\BackwardsCompatibilityTrait;
use Symfony\Component\Yaml\Yaml;

/**
 * @coversDefaultClass \Drupal\track_usage\Recorder
 * @group track_usage
 */
class RecorderTest extends TrackUsageTestBase {

  use BackwardsCompatibilityTrait;

  /**
   * Expectations cache.
   *
   * @var array<array-key, list<string>>
   */
  protected array $expectations;

  /**
   * @covers ::record
   */
  public function testRecorder(): void {
    ConfigurableLanguage::createFromLangcode('it')->save();

    $config = TrackConfig::create([
      'id' => 'test',
      'label' => $this->randomString(),
      'source' => ['node' => []],
      'traversable' => ['media' => [], 'paragraph' => []],
      'target' => ['file' => []],
      'realTimeRecording' => TRUE,
      'activeRevision' => FALSE,
    ]);
    $config->save();

    $this->createFiles();
    $this->createData();
    $node = $this->entity['node'][1];
    $nodeStorage = \Drupal::entityTypeManager()->getStorage('node');
    $paragraphStorage = \Drupal::entityTypeManager()->getStorage('paragraph');

    $this->assertRecordedUsage('Initial content');

    $node->set('title', $this->randomString())->setNewRevision();
    $node->save();
    $this->assertRecordedUsage('NID1: Added revision VID=2');

    File::load(4)->delete();
    $this->assertRecordedUsage('Deleted file FID=4');

    Media::load(2)->delete();
    $this->assertRecordedUsage('Deleted media MID=2');

    $node->get('top_paragraph_field')->entity
      ->get('nested_paragraph_field')->entity
      ->set('text', 'foo')
      ->save();
    $this->assertRecordedUsage('Extracted FID=4 and FID=5 from paragraph text');

    $node->removeTranslation('ro');
    $node->save();
    $this->assertRecordedUsage('NID1: Deleted translation ID=ro');

    $node->addTranslation('it', ['title' => $this->randomString()] + $this->entity['node'][1]->toArray());
    $node->save();
    $this->assertRecordedUsage('NID1: Added translation ID=it');

    ConfigurableLanguage::load('it')->delete();
    $this->assertRecordedUsage('Deleted language ID=it');

    $nodeStorage->deleteRevision(1);
    $this->assertRecordedUsage('NID1: Deleted revision VID=1');

    $node->set('title', $this->randomString())->setNewRevision();
    $node->save();
    $this->assertRecordedUsage('NID1: Added revision VID=3');

    // Try to edit a revision and delete it in the same request.
    $nodeStorage->loadRevision(2)->set('title', $this->randomString())->save();
    $nodeStorage->deleteRevision(2);
    $this->assertRecordedUsage('NID1: Deleted revision VID=2');

    // Delete a paragraph which is a traversable entity. Prove that deleting a
    // traversable entity revision will recalculate source entity usages.
    assert($paragraphStorage instanceof RevisionableStorageInterface);
    $paragraphStorage->deleteRevision(6);
    $this->assertRecordedUsage('Deleted paragraph ID=2, REVISION_ID=6');

    $node->delete();
    $this->assertRecordedUsage('Deleted NID=1');
  }

  /**
   * Asserts that a usage has been recorded.
   *
   * @param string $expectation
   *   The expectation identifier according to recorder.yml test cases.
   */
  protected function assertRecordedUsage(string $expectation): void {
    // Check main usage records.
    $actualRecords = $this->container->get('database')
      ->select(RecorderInterface::TABLE)
      ->fields(RecorderInterface::TABLE)
      ->execute()
      ->fetchAll($this->fetchMode(\PDO::FETCH_NUM));

    $expectedRecords = $this->getExpectation($expectation);
    $this->assertCount(count($expectedRecords), $actualRecords);
    foreach ($actualRecords as $delta => $actualRecord) {
      $tid = (int) array_shift($actualRecord);
      $actualRecord[] = $this->getPaths($tid);
      $this->assertEquals($expectedRecords[$delta], $actualRecord, "Expectation '$expectation' failed at row $delta");
    }
  }

  /**
   * Gets the paths for a given usage.
   *
   * @param int $tid
   *   The usage record ID.
   *
   * @return array<int, array<int, string>>
   *   A structured array of paths.
   */
  public function getPaths(int $tid): array {
    $paths = [];
    $items = $this->container->get('database')->select(RecorderInterface::TABLE_PATHS)
      ->fields(RecorderInterface::TABLE_PATHS)
      ->condition('tid', $tid)
      ->orderBy('path')
      ->orderBy('delta')
      ->execute()
      ->fetchAll($this->fetchMode(\PDO::FETCH_ASSOC));

    foreach ($items as $item) {
      $paths[$item['path']][$item['delta']] = "{$item['type']}:{$item['id']}:{$item['revision']}";
    }

    return $paths;
  }

  /**
   * Processes one expectation case give its identifier.
   *
   * @param string $case
   *   The expectation identifier according to recorder.yml test cases.
   */
  protected function getExpectation(string $case): array {
    if (!isset($this->expectations)) {
      $this->expectations = [];
      $fixture = file_get_contents(__DIR__ . '/../../fixtures/expectation/recorder.yml');
      foreach (Yaml::parse($fixture)['asserts'] as $expectation => $markdown) {
        $lines = explode("\n", trim($markdown));
        $this->expectations[$expectation] = [];
        foreach ($lines as $line) {
          $line = trim($line);
          if (!isset($line[0]) || $line[0] === '#') {
            // Comment.
            continue;
          }
          $columns = array_filter(array_map('trim', explode('|', trim($line, '|'))));
          // Decode the paths-JSON for easy comparison.
          $columns[9] = json_decode($columns[9], TRUE);
          $this->expectations[$expectation][] = $columns;
        }
        $this->expectations[$expectation] = array_filter($this->expectations[$expectation]);
      }
    }
    return $this->expectations[$case];
  }

}
