<?php

declare(strict_types=1);

namespace Drupal\Tests\track_usage\Kernel\Track;

use Drupal\block_content\Entity\BlockContent;
use Drupal\block_content\Entity\BlockContentType;
use Drupal\entity_browser\Entity\EntityBrowser;
use Drupal\entity_test\Entity\EntityTest;
use Drupal\layout_builder\Entity\LayoutBuilderEntityViewDisplay;
use Drupal\layout_builder\Section;
use Drupal\layout_builder\SectionComponent;
use Drupal\layout_builder\SectionListInterface;

/**
 * Tests the 'layout_builder' track plugin.
 *
 * @group track_usage
 */
class LayoutBuilderTest extends TrackTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'block_content',
    'entity_browser',
    'entity_browser_block',
    'entity_browser_entity_form',
    'layout_builder',
    'layout_discovery',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('entity_test');
    $this->installEntitySchema('entity_browser');
    $this->installEntitySchema('block_content');
    BlockContentType::create(['id' => 'basic', 'label' => 'Basic'])->save();

    LayoutBuilderEntityViewDisplay::create([
      'targetEntityType' => 'entity_test',
      'bundle' => 'entity_test',
      'mode' => 'default',
      'status' => TRUE,
    ])->enableLayoutBuilder()->setOverridable()->save();

    EntityBrowser::create([
      'name' => 'test',
      'display' => 'modal',
      'selection_display' => 'no_display',
      'widget_selector' => 'tabs',
      'widgets' => [
        $this->container->get('uuid')->generate() => [
          'id' => 'entity_form',
          'settings' => [
            'entity_type' => 'entity_test',
            'bundle' => 'entity_test',
          ],
        ],
      ],
    ])->save();
  }

  /**
   * @covers \Drupal\track_usage\Plugin\TrackUsage\Track\LayoutBuilder
   */
  public function testPlugin(): void {
    $entity1 = EntityTest::create([
      'type' => 'entity_test',
      'name' => $this->randomString(),
    ]);
    $entity1->save();
    $entity2 = EntityTest::create([
      'type' => 'entity_test',
      'name' => $this->randomString(),
    ]);
    $entity2->save();

    $block1 = BlockContent::create(['info' => 'Foo', 'type' => 'basic']);
    $block1->save();
    $block2 = BlockContent::create(['info' => 'Baz', 'type' => 'basic']);
    $block2->save();

    $entity = EntityTest::create([
      'type' => 'entity_test',
      'name' => $this->randomString(),
    ]);

    $field = $entity->get('layout_builder__layout');
    assert($field instanceof SectionListInterface);

    $uuid = $this->container->get('uuid')->generate();
    $field->appendSection(new Section('layout_onecol', [], [
      $uuid => new SectionComponent($uuid, 'content', [
        'id' => 'entity_browser_block:test',
        'entity_ids' => [
          "entity_test:{$entity1->id()}",
          "entity_test:{$entity2->id()}",
        ],
        'view_modes' => [
          "entity_test:{$entity1->id()}" => 'full',
          "entity_test:{$entity2->id()}" => 'full',
        ],

      ]),
    ])
    );

    $uuid1 = $this->container->get('uuid')->generate();
    $uuid2 = $this->container->get('uuid')->generate();
    $field->appendSection(new Section('layout_onecol', [], [
      $uuid1 => new SectionComponent($uuid1, 'content', [
        'id' => 'inline_block:basic',
        'block_id' => $block1->id(),
        'block_revision_id' => $block1->getRevisionId(),
      ]),
      $uuid2 => new SectionComponent($uuid2, 'content', [
        'id' => "block_content:{$block2->uuid()}",
      ]),
    ])
    );
    $entity->save();

    $this->assertTargetEntities('layout_builder', $entity, 'layout_builder__layout', [
      $entity1,
      $entity2,
      $block1,
      $block2,
    ]);
  }

}
