<?php

declare(strict_types=1);

namespace Drupal\Tests\track_usage\Kernel;

use Drupal\track_usage\Entity\TrackConfig;
use Symfony\Component\Yaml\Yaml;

/**
 * @coversDefaultClass \Drupal\track_usage\Tracker
 * @group track_usage
 */
class TrackerTest extends TrackUsageTestBase {

  /**
   * @covers \Drupal\track_usage\Tracker
   */
  public function testTracker(): void {
    $this->createFiles();
    $this->createData();

    $tracker = $this->container->get('track_usage.tracker');
    foreach ($this->getTestCases() as $delta => $case) {
      $config = TrackConfig::load('default');
      $actualUsages = $tracker->track($case['entity'], $config);
      $this->assertEqualsCanonicalizing($case['expected'], $actualUsages->toArray(), "Case $delta");
      $config->delete();
    }
  }

  /**
   * Provides test cases for ::testTracker().
   *
   * We're not using data provider because we want to only create testing data
   * once and apply successive transformation.
   *
   * @return iterable
   *   Test cases.
   */
  protected function getTestCases(): iterable {
    $entityTypeManager = $this->container->get('entity_type.manager');

    $fixture = file_get_contents(__DIR__ . '/../../fixtures/expectation/tracker.yml');
    $cases = Yaml::parse($fixture)['cases'];
    foreach ($cases as $case) {
      $case['entity_data'] = array_pad($case['entity_data'], 2, []);
      $case['transformation_step'] ??= NULL;

      // First element is the starting entity.
      [$entityTypeId, $id, $langcode] = array_pad(explode(':', $case['entity_data'][0]), 3, NULL);
      $entity = $entityTypeManager->getStorage($entityTypeId)->load($id);
      if ($langcode) {
        $entity = $entity->getTranslation($langcode);
      }

      // The second element is the path to a referred entity.
      foreach ($case['entity_data'][1] as $field) {
        [$fieldName, $delta] = explode(':', $field);
        $entity = $entity->get($fieldName)->get($delta)->entity;
      }
      $case['entity'] = $entity;

      TrackConfig::create([
        'id' => 'default',
        'label' => 'Default',
      ] + $case['config'])->save();

      if ($case['transformation_step']) {
        $this->transformContent($case['transformation_step']);
      }

      yield $case;
    }
  }

  /**
   * Applies transformations to content in order to test different scenarios.
   *
   * @param string $transformationStep
   *   The transformation step to be applied.
   */
  protected function transformContent(string $transformationStep): void {
    switch ($transformationStep) {
      case 'add_revision':
        $this->entity['node'][1]->set('title', $this->randomString());
        $this->entity['node'][1]->setNewRevision();
        $this->entity['node'][1]->save();
        break;

      case 'delete_file_fid4':
        $this->file[4]->delete();
        break;

      case 'delete_media_mid2':
        $this->entity['node'][1]->get('media_file')->entity->delete();
        break;

      case 'remove_embedded':
        $this->entity['node'][1]
          ->get('top_paragraph_field')->entity
          ->get('nested_paragraph_field')->entity
          ->set('text', 'foo')
          ->save();
    }
  }

}
