<?php

declare(strict_types=1);

namespace Drupal\track_usage\Form;

use Drupal\Component\Render\MarkupInterface;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\track_usage\Entity\TrackConfig;
use Drupal\track_usage\Entity\TrackConfigInterface;
use Drupal\track_usage\Model\EntityRole;
use Drupal\track_usage\Plugin\TrackUsage\TrackPluginManagerInterface;

/**
 * Track config entity form.
 */
class TrackConfigForm extends EntityForm {

  use AutowireTrait;

  private const ALL = '__all_bundles__';

  public function __construct(
    protected TrackPluginManagerInterface $trackPluginManager,
    protected EntityTypeBundleInfoInterface $entityTypeBundleInfo,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): array {
    $config = $this->getEntity();

    $form['warning'] = [
      '#theme' => 'status_messages',
      '#message_list' => [
        'warning' => [
          $this->t('Any change to the configuration might require a full rebuild of tracking data.'),
        ],
      ],
      '#status_headings' => [
        'warning' => $this->t('Warning message'),
      ],
    ];

    $form['label'] = [
      '#title' => $this->t('Label'),
      '#type' => 'textfield',
      '#default_value' => $config->label(),
      '#description' => $this->t('Choose an administrative label for this configuration.'),
      '#required' => TRUE,
      '#size' => 30,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $config->id(),
      '#maxlength' => 32,
      '#disabled' => !$config->isNew(),
      '#machine_name' => [
        'exists' => [TrackConfig::class, 'load'],
        'source' => ['label'],
      ],
    ];

    $form['trackPlugins'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Track plugins'),
      '#description' => $this->t('If none is selected, all plugins will be used.'),
      '#options' => array_map(
        fn(array $definition): string|MarkupInterface => $definition['label'],
        $this->trackPluginManager->getDefinitions(),
      ),
      '#default_value' => $config->getTrackPluginIds(),
    ];

    $form['activeRevision'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Track only the active entity revision'),
      '#default_value' => $config->trackOnlyActiveRevision(),
    ];

    $form['realTimeRecording'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Record entity changes in real time'),
      '#default_value' => $config->useRealTimeRecording(),
    ];

    foreach (EntityRole::cases() as $role) {
      $form[$role->value] = [
        '#type' => 'details',
        '#title' => $role->getTitle(),
        '#description' => $role->getDescription(),
        '#tree' => TRUE,
      ];
      foreach ($this->getEntityDefinitions($role) as $entityType) {
        $this->buildEntityRoleForm($form, $role, $entityType);
      }
    }

    $form['status'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Active'),
      '#default_value' => $config->status(),
    ];

    return parent::form($form, $form_state);
  }

  /**
   * Builds the entity role form.
   *
   * @param array $form
   *   The form render array passed by reference.
   * @param \Drupal\track_usage\Model\EntityRole $role
   *   The entity role.
   * @param \Drupal\Core\Entity\EntityTypeInterface $entityType
   *   The entity type definition.
   */
  protected function buildEntityRoleForm(array &$form, EntityRole $role, EntityTypeInterface $entityType): void {
    $selector = ":input[name='$role->value[{$entityType->id()}][" . self::ALL . "]']";
    $hasBundleKey = $entityType->hasKey('bundle');

    $config = $this->getEntity()->get($role->value);
    $defaultValue = $config[$entityType->id()] ?? [];
    if (array_key_exists($entityType->id(), $config) && !$defaultValue) {
      // This is the case when all bundles of an entity type needed.
      $defaultValue = [self::ALL];
    }

    $form[$role->value][$entityType->id()] = [
      '#type' => 'checkboxes',
      '#title' => $entityType->getLabel(),
      '#options' => [
        self::ALL => $hasBundleKey ? $this->t('All bundles') : $entityType->getLabel(),
      ] + $this->getBundleLabels($entityType->id()),
      '#default_value' => $defaultValue,
    ];

    if (!$hasBundleKey) {
      $form[$role->value][$entityType->id()][$entityType->id()]['#access'] = FALSE;
    }
    foreach ($this->getBundleLabels($entityType->id()) as $bundle => $bundleLabel) {
      $form[$role->value][$entityType->id()][$bundle]['#states'] = [
        'visible' => [$selector => ['checked' => FALSE]],
      ];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    // Normalize values.
    $form_state->setValue('trackPlugins', array_keys(array_filter($form_state->getValue('trackPlugins'))));
    foreach (EntityRole::cases() as $role) {
      foreach ($form_state->getValue($role->value) as $entityTypeId => $bundles) {
        $bundles = array_filter($bundles);
        if (isset($bundles[self::ALL])) {
          $form_state->setValue([$role->value, $entityTypeId], []);
        }
        elseif ($bundles) {
          $form_state->setValue([$role->value, $entityTypeId], array_keys($bundles));
        }
        else {
          $form_state->unsetValue([$role->value, $entityTypeId]);
        }
      }
    }

    if (empty($form_state->getValue(EntityRole::Source->value))) {
      $form_state->setErrorByName(EntityRole::Source->value, $this->t('You must select at least one entity type as source'));
    }
    if (empty($form_state->getValue(EntityRole::Target->value))) {
      $form_state->setErrorByName(EntityRole::Target->value, $this->t('You must select at least one entity type as target'));
    }

    parent::validateForm($form, $form_state);
    $form_state->setRedirect('entity.track_usage_config.collection');
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config_name = $this->getEntity()->getConfigDependencyName();
    $original = $this->configFactory()->getEditable($config_name);
    $original = json_encode($original->getRawData());
    parent::submitForm($form, $form_state);
    $updated = clone $this->getEntity();
    $updated = $updated->toArray();
    $updated['status'] = (bool) $updated['status'];
    $updated = json_encode($updated);

    if ($original !== $updated) {
      $link = Link::createFromRoute('rebuild', 'track_usage.settings_form')->toString();
      $this->messenger()->addWarning($this->t('Tracking data needs to be @rebuild.', [
        '@rebuild' => $link,
      ]));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getEntity(): TrackConfigInterface {
    $config = parent::getEntity();
    assert($config instanceof TrackConfigInterface);
    return $config;
  }

  /**
   * Returns a list of entity types suitable for the given entity role.
   *
   * @param \Drupal\track_usage\Model\EntityRole $role
   *   The entity role.
   *
   * @return array<array-key, \Drupal\Core\Entity\EntityTypeInterface>
   *   A list of entity types suitable for the given entity role.
   */
  protected function getEntityDefinitions(EntityRole $role): iterable {
    $definitions = $this->entityTypeManager->getDefinitions();

    if ($role->onlyFieldableEntityTypes()) {
      $definitions = array_filter(
        $definitions,
        fn(EntityTypeInterface $entityType): bool => $entityType->entityClassImplements(FieldableEntityInterface::class),
      );
    }

    uasort($definitions, function (EntityTypeInterface $a, EntityTypeInterface $b): int {
      return (string) $a->getLabel() <=> (string) $b->getLabel();
    });

    foreach ($definitions as $entityTypeId => $definition) {
      yield $entityTypeId => $definition;
    }
  }

  /**
   * Returns a sorted list of bundle labels given an entity type.
   *
   * @param string $entityTypeId
   *   The ID of the entity type.
   *
   * @return array
   *   Associative array of bundle labels keyed by bundle.
   */
  protected function getBundleLabels(string $entityTypeId): array {
    $labels = array_map(
      fn(array $info) => $info['label'],
      $this->entityTypeBundleInfo->getBundleInfo($entityTypeId),
    );
    asort($labels);
    return $labels;
  }

}
