<?php

declare(strict_types=1);

namespace Drupal\track_usage\Plugin\QueueWorker;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\Attribute\QueueWorker;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\track_usage\Entity\TrackConfigInterface;
use Drupal\track_usage\RecorderInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Updates usages.
 */
#[QueueWorker(
  id: 'track_usage',
  title: new TranslatableMarkup('Track Usage'),
  cron: ['time' => 30],
)]
class TrackUsageWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly RecorderInterface $recorder,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get(RecorderInterface::class),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($data): void {
    if (empty($data['type']) || empty($data['ids']) || !array_key_exists('revision', $data) || empty($data['config'])) {
      // Malformed item.
      return;
    }

    $config = $this->entityTypeManager->getStorage('track_usage_config')->load($data['config']);
    if (!$config instanceof TrackConfigInterface) {
      return;
    }

    $storage = $this->entityTypeManager->getStorage($data['type']);
    $method = $data['revision'] ? 'loadMultipleRevisions' : 'loadMultiple';
    foreach ($storage->{$method}($data['ids']) as $entity) {
      $this->recorder->record($entity, $config);
    }
  }

}
