<?php

declare(strict_types=1);

namespace Drupal\track_usage\Plugin\TrackUsage;

use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\FieldItemInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\PluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base class for file track usage plugins.
 */
abstract class TrackPluginBase extends PluginBase implements TrackPluginInterface, ContainerFactoryPluginInterface {

  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    protected readonly EntityRepositoryInterface $entityRepository,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity.repository'),
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getTargetEntities(FieldItemListInterface $itemList): iterable {
    $yielded = [];
    foreach ($itemList as $item) {
      foreach ($this->getItemTargetEntities($item) as $entity) {
        $key = "{$entity->getEntityTypeId()}:{$entity->id()}";
        if (!isset($yielded[$key])) {
          // Prevent duplicates.
          $yielded[$key] = TRUE;
          yield $entity;
        }
      }
    }
  }

  /**
   * Retrieves the target entities from a field item.
   *
   * @param \Drupal\Core\Field\FieldItemInterface $item
   *   The field item instance.
   *
   * @return iterable<\Drupal\Core\Entity\EntityInterface>
   *   A list of target entities.
   */
  abstract protected function getItemTargetEntities(FieldItemInterface $item): iterable;

}
