<?php

declare(strict_types=1);

namespace Drupal\track_usage\Trait;

use Drupal\Core\Config\Entity\ConfigEntityInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\track_usage\Model\EntityRole;

/**
 * Reusable code that deals with ID columns casting.
 */
trait IdColumnTrait {

  /**
   * Static cache for the detected ID column type.
   *
   * @var array<array-key, bool>
   */
  protected array $idColumnIsInteger = [];

  /**
   * Checks whether the entity type ID is an integer.
   *
   * @param string $entityTypeId
   *   The entity type ID.
   *
   * @return bool
   *   Whether the entity type ID is an integer.
   */
  protected function isIdColumnInteger(string $entityTypeId): bool {
    if (!isset($this->idColumnIsInteger[$entityTypeId])) {
      $entityType = $this->entityTypeManager->getDefinition($entityTypeId);
      if (!$entityType->hasKey('id')) {
        throw new \InvalidArgumentException("Entity type '$entityTypeId' has no ID key and cannot be used");
      }
      if ($entityType->entityClassImplements(ConfigEntityInterface::class)) {
        $this->idColumnIsInteger[$entityTypeId] = FALSE;
      }
      elseif ($entityType->entityClassImplements(FieldableEntityInterface::class)) {
        $fieldDefinition = $this->entityFieldManager->getBaseFieldDefinitions($entityTypeId)[$entityType->getKey('id')];
        $this->idColumnIsInteger[$entityTypeId] = $fieldDefinition->getType() === 'integer';
      }
      else {
        // Should not happen. For any case fallback to string which is safe.
        $this->idColumnIsInteger[$entityTypeId] = FALSE;
      }
    }
    return $this->idColumnIsInteger[$entityTypeId];
  }

  /**
   * Returns the source or target entity ID column name.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface|string $entityOrEntityTypeId
   *   Either the entity or the entity type ID.
   * @param \Drupal\track_usage\Model\EntityRole $role
   *   The entity role.
   *
   * @return string
   *   The source or target entity ID column name.
   */
  protected function getIdColumnName(EntityInterface|string $entityOrEntityTypeId, EntityRole $role): string {
    assert(!$role->isTraversable());
    $entityTypeId = is_string($entityOrEntityTypeId) ? $entityOrEntityTypeId : $entityOrEntityTypeId->getEntityTypeId();
    return $this->isIdColumnInteger($entityTypeId) ? "{$role->value}_id_int" : "{$role->value}_id_string";
  }

  /**
   * Returns the value of ID to be stored in the source/target_id_int column.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   *
   * @return int
   *   The value to be stored in the source/target_id_int column.
   */
  protected function getIntegerIdColumnValue(EntityInterface $entity): int {
    return $this->isIdColumnInteger($entity->getEntityTypeId()) ? (int) $entity->id() : 0;
  }

  /**
   * Returns the value of ID to be stored in the source/target_id_string column.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   *
   * @return string
   *   The value to be stored in the source/target_id_string column.
   */
  protected function getStringIdColumnValue(EntityInterface $entity): string {
    return (string) $entity->id();
  }

  /**
   * Casts the entity ID value given an entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   *
   * @return int|string
   *   The entity ID.
   */
  protected function getIdColumnValueByEntity(EntityInterface $entity): int|string {
    return $this->getIdColumnValue($entity->getEntityTypeId(), $entity->id());
  }

  /**
   * Casts the entity ID value given the entity type and the entity ID.
   *
   * @param string $entityTypeId
   *   The entity type ID.
   * @param int|string $entityId
   *   The entity ID.
   *
   * @return int|string
   *   The entity ID.
   */
  protected function getIdColumnValue(string $entityTypeId, int|string $entityId): int|string {
    return $this->isIdColumnInteger($entityTypeId) ? (int) $entityId : (string) $entityId;
  }

}
