<?php

declare(strict_types=1);

namespace Drupal\track_usage;

use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\track_usage\Entity\TrackConfigInterface;
use Drupal\track_usage\Model\BulkUpdateMethod;
use Drupal\track_usage\Model\Operation;
use Drupal\track_usage\Trait\EntityUtilityTrait;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Bulk record all usages.
 */
class Updater implements UpdaterInterface {

  use EntityUtilityTrait;
  use StringTranslationTrait;

  public function __construct(
    protected readonly ConfigFactoryInterface $configFactory,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly Connection $db,
    protected readonly RecorderInterface $recorder,
    #[Autowire(service: 'queue')]
    protected readonly QueueFactory $queue,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function update(BulkUpdateMethod $bulkUpdateMethod, TrackConfigInterface $config): void {
    $this->recorder->cleanup($config, Operation::Delete);
    $method = $bulkUpdateMethod->value;
    $this->$method($config);
  }

  /**
   * Runs update as a batch process.
   *
   * @param \Drupal\track_usage\Entity\TrackConfigInterface $config
   *   The track config.
   */
  protected function batch(TrackConfigInterface $config): void {
    $batchBuild = (new BatchBuilder())
      ->setTitle($this->t('Update usages'))
      ->addOperation([static::class, 'initBatch']);
    foreach ($this->getEntities($config) as $entityTypeId => $ids) {
      foreach (array_chunk($ids, 50) as $chunk) {
        $batchBuild->addOperation([static::class, 'recordUsages'], [$entityTypeId, $chunk, $config->id()]);
      }
    }
    batch_set($batchBuild->toArray());
  }

  /**
   * Runs update using the queue.
   *
   * @param \Drupal\track_usage\Entity\TrackConfigInterface $config
   *   The track config.
   */
  protected function queue(TrackConfigInterface $config): void {
    $queue = $this->queue->get('track_usage');
    $queue->createQueue();

    foreach ($this->getEntities($config) as $entityTypeId => $ids) {
      $isRevisionable = $this->isRevisionable($entityTypeId) && !$config->trackOnlyActiveRevision();
      foreach (array_chunk($ids, 10) as $chunk) {
        $queue->createItem([
          'type' => $entityTypeId,
          'ids' => $chunk,
          'revision' => $isRevisionable,
          'config' => $config->id(),
        ]);
      }
    }
  }

  /**
   * Runs update using the queue.
   *
   * @param \Drupal\track_usage\Entity\TrackConfigInterface $config
   *   The track config.
   */
  protected function instant(TrackConfigInterface $config): void {
    foreach ($this->getEntities($config) as $entityTypeId => $ids) {
      $isRevisionable = $this->isRevisionable($entityTypeId) && !$config->trackOnlyActiveRevision();
      $method = $isRevisionable ? 'loadMultipleRevisions' : 'loadMultiple';
      $storage = $this->entityTypeManager->getStorage($entityTypeId);
      foreach (array_chunk($ids, 10) as $chunk) {
        foreach ($storage->{$method}($chunk) as $entity) {
          $this->recorder->record($entity, $config);
        }
      }
    }
  }

  /**
   * Returns list of entities whose usages should be updated.
   *
   * @param \Drupal\track_usage\Entity\TrackConfigInterface $config
   *   The track config.
   *
   * @return iterable<array-key, list<int|string>>
   *   Associative array keyed by entity type, having lists of IDs as values.
   */
  protected function getEntities(TrackConfigInterface $config): iterable {
    foreach ($config->get('source') as $entityTypeId => $bundles) {
      $isRevisionable = $this->isRevisionable($entityTypeId) && !$config->trackOnlyActiveRevision();

      $query = $this->entityTypeManager->getStorage($entityTypeId)->getQuery()->accessCheck(FALSE);
      if ($isRevisionable) {
        $query->allRevisions();
      }
      if ($bundles && ($bundleKey = $this->entityTypeManager->getDefinition($entityTypeId)->getKey('bundle'))) {
        $query->condition($bundleKey, $bundles, 'IN');
      }

      if ($ids = $query->execute()) {
        $ids = $isRevisionable ? array_keys($ids) : array_values($ids);
        yield $entityTypeId => $ids;
      }
    }
  }

  /**
   * Initializes the batch process.
   *
   * @param array $context
   *   The batch process context.
   */
  public static function initBatch(array &$context): void {
    $context['results']['progress']['source'] = 0;
    $context['results']['progress']['target'] = 0;
  }

  /**
   * Processes one entity/revision at a time.
   *
   * @param string $entityTypeId
   *   The entity type ID.
   * @param array $ids
   *   Entity IDs.
   * @param string $configId
   *   The track config ID.
   * @param array $context
   *   Batch process context.
   */
  public static function recordUsages(string $entityTypeId, array $ids, string $configId, array &$context): void {
    $entityTypeManager = \Drupal::entityTypeManager();
    $recorder = \Drupal::service(Recorder::class);

    $config = $entityTypeManager->getStorage('track_usage_config')->load($configId);
    if (!$config instanceof TrackConfigInterface) {
      // It might have been deleted in the meantime?
      $context['finished'] = 1;
      return;
    }

    if (!isset($context['sandbox']['ids'])) {
      $context['sandbox']['ids'] = $ids;
      $context['sandbox']['progress'] = 0;
      $context['sandbox']['total'] = count($ids);
      $context['sandbox']['load_method'] = (static::isRevisionable($entityTypeId) && !$config->trackOnlyActiveRevision()) ? 'loadRevision' : 'load';
    }

    $id = array_shift($context['sandbox']['ids']);
    $context['sandbox']['progress']++;
    $context['results']['progress']['source']++;

    $storage = $entityTypeManager->getStorage($entityTypeId);
    if ($entity = $storage->{$context['sandbox']['load_method']}($id)) {
      $recorder->record($entity, $config);
    }

    if ($context['sandbox']['progress'] < $context['sandbox']['total']) {
      $context['finished'] = $context['sandbox']['progress'] / $context['sandbox']['total'];
    }
    else {
      $context['finished'] = 1;
      $context['message'] = t('Processed @count source entities', [
        '@count' => $context['results']['progress']['source'],
      ]);
    }
  }

}
