<?php

declare(strict_types=1);

namespace Drupal\Tests\track_usage\Kernel\Track;

use Drupal\Core\Entity\EntityInterface;
use Drupal\entity_test\Entity\EntityTest;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\file\Entity\File;
use Drupal\file\FileInterface;
use Drupal\filter\Entity\FilterFormat;
use Drupal\language\Entity\ConfigurableLanguage;
use Drupal\node\Entity\NodeType;
use Drupal\redirect\Entity\Redirect;
use Drupal\Tests\node\Traits\NodeCreationTrait;
use Drupal\Tests\TestFileCreationTrait;

/**
 * Tests the 'html_link' track plugin.
 *
 * @group track_usage
 */
class HtmlLinkTest extends TrackTestBase {

  use NodeCreationTrait;
  use TestFileCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'filter',
    'language',
    'node',
    'path',
    'search',
    'text',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('entity_test');
    $this->installEntitySchema('file');
    $this->installEntitySchema('node');
    $this->installEntitySchema('redirect');
    $this->installEntitySchema('user');
    $this->installEntitySchema('path_alias');
    $this->installSchema('file', ['file_usage']);
    $this->installConfig(['language']);

    ConfigurableLanguage::createFromLangcode('fr')->save();
    \Drupal::service('kernel')->rebuildContainer();
    // For some reason, after rebuilding the container wrappers are lost.
    $this->container->get('stream_wrapper_manager')->register();

    FieldStorageConfig::create([
      'field_name' => 'text',
      'type' => 'text_with_summary',
      'entity_type' => 'entity_test',
      'cardinality' => 1,
    ])->save();
    FieldConfig::create([
      'field_name' => 'text',
      'label' => $this->randomString(),
      'entity_type' => 'entity_test',
      'bundle' => 'entity_test',
    ])->save();
    FilterFormat::create([
      'format' => 'basic_html',
      'name' => 'Basic HTML',
    ])->setFilterConfig('filter_html', [
      'status' => TRUE,
      'settings' => [
        'allowed_html' => '<a href>',
      ],
    ])->setFilterConfig('filter_url', [
      'status' => TRUE,
      'settings' => [
        'filter_url_length' => 30,
      ],
    ])->save();
    NodeType::create(['type' => 'page', 'name' => 'Page'])->save();
  }

  /**
   * @covers \Drupal\track_usage\Plugin\TrackUsage\Track\HtmlLink
   */
  public function testPlugin(): void {
    $ref1 = $this->createNode([
      'type' => 'page',
      'title' => $this->randomString(),
      'path' => ['alias' => '/test/aliased/relative'],
    ]);
    $ref2 = $this->createNode([
      'type' => 'page',
      'title' => $this->randomString(),
      'path' => ['alias' => '/test/aliased/absolute'],
    ]);
    $ref3 = $this->createNode([
      'type' => 'page',
      'title' => $this->randomString(),
    ]);
    $ref4 = $this->createNode([
      'type' => 'page',
      'title' => $this->randomString(),
    ]);
    $ref5 = $this->createNode([
      'type' => 'page',
      'title' => $this->randomString(),
      'langcode' => 'fr',
    ]);
    $ref6 = $this->createNode([
      'type' => 'page',
      'title' => $this->randomString(),
    ]);
    $redirect = Redirect::create();
    $redirect->setSource('this/is/a/redirect');
    $redirect->setRedirect($ref5->toUrl()->toString());
    $redirect->setLanguage('fr');
    $redirect->save();

    $files = $this->getTestFiles('image');

    $file1 = File::create(['uri' => $files[0]->uri]);
    $file1->save();
    $file2 = File::create(['uri' => $files[1]->uri]);
    $file2->save();
    $file3 = File::create(['uri' => $files[2]->uri]);
    $file3->save();

    $text = $this->randomString() .
      $this->buildEntityAnchor(entity: $ref1) . ' ' . $this->randomString() . ' ' .
      $this->buildEntityAnchor(entity: $ref2, absolute: TRUE) . ' ' . $this->randomString() . ' ' .
      $this->buildEntityAnchor(entity: $ref3) . ' ' . $this->randomString() . ' ' .
      $this->buildEntityAnchor(entity: $ref4, absolute: TRUE) . ' ' . $this->randomString() . ' ' .
      $this->buildFileAnchor($file1) . ' ' . $this->randomString() . ' ' .
      $this->buildFileAnchor($file2, TRUE) . ' ' . $this->randomString() . ' ' .
      '<a href="/search?q=abc">Search</a> ' . $this->randomString() . ' ' .
      '<a href="/fr/this/is/a/redirect">Redirect</a> ' . $this->randomString() . ' ' .
      '<a href="http://example.com">example.com</a> ' . $this->randomString() . ' ' .
      '<a href="example@example.com">example@example.com</a> ' . $this->randomString() . ' ' .
      '<a href="mailto:example@example.com">example@example.com</a> ' . $this->randomString() . ' ' .
      $ref6->toUrl()->setAbsolute()->toString() . ' ' . $this->randomString();
    $summary = $this->randomString() . $this->buildFileAnchor($file3) . $this->randomString();

    $entity = EntityTest::create([
      'type' => 'entity_test',
      'name' => $this->randomString(),
      'text' => [
        'value' => $text,
        'summary' => $summary,
        'format' => 'basic_html',
      ],
    ]);
    $entity->save();

    $this->assertTargetEntities('html_link', $entity, 'text', [
      $ref1,
      $ref2,
      $ref3,
      $ref4,
      $ref5,
      $ref6,
      $file1,
      $file2,
      $file3,
    ]);
  }

  /**
   * Builds an HTML hyperlink to an entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   * @param bool $absolute
   *   (optional) Whether the URL should be absolute. Defaults to FALSE.
   *
   * @return string
   *   HTML tag.
   */
  protected function buildEntityAnchor(EntityInterface $entity, bool $absolute = FALSE): string {
    $url = $entity->toUrl();
    if ($absolute) {
      $url->setAbsolute();
    }
    return '<a href="' . $url->toString() . '">' . $this->randomString() . '</a>';
  }

  /**
   * Builds an HTML hyperlink to a file.
   *
   * @param \Drupal\file\FileInterface $file
   *   The file entity.
   * @param bool $absolute
   *   (optional) Whether the URL should be absolute. Defaults to FALSE.
   *
   * @return string
   *   HTML tag.
   */
  protected function buildFileAnchor(FileInterface $file, bool $absolute = FALSE): string {
    $fileUrlGenerator = $this->container->get('file_url_generator');
    $href = $fileUrlGenerator->generate($file->getFileUri());
    if ($absolute) {
      $href->setAbsolute();
    }
    return '<a href="' . $href->toString() . '">' . $this->randomString() . '</a>';
  }

}
