<?php

declare(strict_types=1);

namespace Drupal\track_usage\Entity;

use Drupal\Core\Config\Entity\ConfigEntityInterface;
use Drupal\Core\Entity\EntityInterface;

/**
 * Track preset entities interface.
 */
interface TrackConfigInterface extends ConfigEntityInterface {

  /**
   * Returns a list of track plugins.
   *
   * @return list<non-empty-string>
   *   List of track plugin IDs.
   */
  public function getTrackPluginIds(): array;

  /**
   * Checks whether to track only the source entity active revision.
   *
   * @return bool
   *   Whether to track only the source entity active revision.
   */
  public function trackOnlyActiveRevision(): bool;

  /**
   * Checks whether to record entity changes in real time.
   *
   * @return bool
   *   Whether to record entity changes in real time.
   */
  public function useRealTimeRecording(): bool;

  /**
   * Checks whether an entity is configured as 'source' entity.
   *
   * @return bool
   *   Whether an entity is configured as 'source' entity.
   */
  public function isSource(EntityInterface $entity): bool;

  /**
   * Checks whether an entity is configured as 'traversable' entity.
   *
   * @return bool
   *   Whether an entity is configured as 'traversable' entity.
   */
  public function isTraversable(EntityInterface $entity): bool;

  /**
   * Checks whether an entity is configured as 'target' entity.
   *
   * @return bool
   *   Whether an entity is configured as 'target' entity.
   */
  public function isTarget(EntityInterface $entity): bool;

  /**
   * Checks whether a given entity type is potentially targetable.
   *
   * A targetable entity type is either traversable or target.
   *
   * @param string $entityTypeId
   *   The entity type ID.
   *
   * @return bool
   *   Whether a given entity type is potentially targetable.
   */
  public function couldBeTargetableEntityType(string $entityTypeId): bool;

  /**
   * Returns the targetable bundles of a given entity type.
   *
   * A targetable entity type bundle is either traversable or target.
   *
   * @param string $entityTypeId
   *   The entity type ID.
   *
   * @return array|null
   *   A list of bundles or NULL if all bundles are targetable.
   */
  public function getTargetableBundles(string $entityTypeId): ?array;

}
