<?php

declare(strict_types=1);

namespace Drupal\track_usage\Plugin\TrackUsage\BlockTarget;

use Drupal\block_content\BlockContentInterface;
use Drupal\block_content\Plugin\Block\BlockContentBlock;
use Drupal\Core\Block\BlockManagerInterface;
use Drupal\Core\Block\BlockPluginInterface;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\track_usage\Attribute\BlockTarget;
use Drupal\track_usage\Plugin\TrackUsage\BlockTargetPluginBase;
use Drupal\track_usage\Plugin\TrackUsage\BlockTargetPluginInterface;
use Drupal\track_usage\Plugin\TrackUsage\BlockTargetPluginManagerInterface;
use Drupal\layout_builder\Plugin\Block\InlineBlock as BlockInlineBlock;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Inline Content block target plugin.
 */
#[BlockTarget(
  id: 'inline_block',
  label: new TranslatableMarkup('Block Content'),
  blockPlugins: ['inline_block'],
)]
class InlineBlock extends BlockTargetPluginBase {

  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    EntityRepositoryInterface $entityRepository,
    EntityTypeManagerInterface $entityTypeManager,
    protected readonly BlockManagerInterface $blockManager,
    protected readonly BlockTargetPluginManagerInterface $blockTargetPluginManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $entityRepository, $entityTypeManager);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity.repository'),
      $container->get('entity_type.manager'),
      $container->get('plugin.manager.block'),
      $container->get(BlockTargetPluginManagerInterface::class),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getTargetEntities(BlockPluginInterface $block): iterable {
    assert($block instanceof BlockInlineBlock);
    if ($revisionId = $block->getConfiguration()['block_revision_id'] ?? NULL) {
      if ($blockContent = $this->entityTypeManager->getStorage('block_content')->loadRevision($revisionId)) {
        assert($blockContent instanceof BlockContentInterface);
        $blockContentPlugin = $this->blockManager->createInstance("block_content:{$blockContent->uuid()}");
        assert($blockContentPlugin instanceof BlockContentBlock);
        $blockContentBlockTarget = $this->blockTargetPluginManager->createInstance('block_content');
        assert($blockContentBlockTarget instanceof BlockTargetPluginInterface);
        foreach ($blockContentBlockTarget->getTargetEntities($blockContentPlugin) as $entity) {
          yield $entity;
        }
      }
    }
  }

}
