<?php

declare(strict_types=1);

namespace Drupal\track_usage\Plugin\TrackUsage\Track;

use Drupal\Core\Block\BlockManagerInterface;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\FieldItemInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\track_usage\Plugin\TrackUsage\BlockTargetPluginManagerInterface;
use Drupal\track_usage\Plugin\TrackUsage\TrackPluginBase;
use Drupal\track_usage\Attribute\Track;
use Drupal\layout_builder\Plugin\Field\FieldType\LayoutSectionItem;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Tracks usage of entities related in Layout Builder layouts.
 */
#[Track(
  id: 'layout_builder',
  label: new TranslatableMarkup('Layout builder'),
  fieldTypes: ['layout_section'],
)]
class LayoutBuilder extends TrackPluginBase {

  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    EntityRepositoryInterface $entityRepository,
    EntityTypeManagerInterface $entityTypeManager,
    protected readonly BlockManagerInterface $blockManager,
    protected readonly BlockTargetPluginManagerInterface $blockTargetPluginManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $entityRepository, $entityTypeManager);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity.repository'),
      $container->get('entity_type.manager'),
      $container->get('plugin.manager.block'),
      $container->get(BlockTargetPluginManagerInterface::class),
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getItemTargetEntities(FieldItemInterface $item): iterable {
    assert($item instanceof LayoutSectionItem);
    foreach ($item->section->getComponents() as $component) {
      $block = $component->getPlugin();
      foreach ($this->blockTargetPluginManager->getApplicablePlugins($block->getBaseId()) as $plugin) {
        foreach ($plugin->getTargetEntities($block) as $entity) {
          yield $entity;
        }
      }

      $entitiesPerType = [];
      foreach ($block->calculateDependencies()['content'] ?? [] as $dependency) {
        [$entityTypeId, , $entityId] = explode(':', $dependency);
        $entitiesPerType[$entityTypeId][] = $entityId;
      }

      foreach ($entitiesPerType as $entityTypeId => $ids) {
        $ids = array_unique($ids);
        if ($this->entityTypeManager->hasDefinition($entityTypeId)) {
          foreach ($this->entityTypeManager->getStorage($entityTypeId)->loadMultiple($ids) as $entity) {
            yield $entity;
          }
        }
      }
    }
  }

}
