<?php

declare(strict_types=1);

namespace Drupal\track_usage\Plugin\TrackUsage\Track;

use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\FieldItemInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\track_usage\EntityGuesser;
use Drupal\track_usage\EntityGuesserInterface;
use Drupal\track_usage\Plugin\TrackUsage\TrackPluginBase;
use Drupal\track_usage\Attribute\Track;
use Drupal\link\LinkItemInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Tracks usage of entities related in Link fields.
 */
#[Track(
  id: 'link',
  label: new TranslatableMarkup('Link field'),
  fieldTypes: ['link', 'link_tree'],
)]
class Link extends TrackPluginBase {

  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    EntityRepositoryInterface $entityRepository,
    EntityTypeManagerInterface $entityTypeManager,
    protected readonly EntityGuesserInterface $entityGuesser,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $entityRepository, $entityTypeManager);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity.repository'),
      $container->get('entity_type.manager'),
      $container->get(EntityGuesser::class),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getItemTargetEntities(FieldItemInterface $item): iterable {
    assert($item instanceof LinkItemInterface);
    if ($entity = $this->entityGuesser->guessFromUrl($item->getUrl()->toUriString())) {
      yield $entity;
    }
  }

}
