<?php

declare(strict_types=1);

namespace Drupal\track_usage;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\track_usage\Entity\TrackConfigInterface;
use Drupal\track_usage\Model\UsageCollection;
use Drupal\track_usage\Plugin\TrackUsage\TrackPluginManagerInterface;
use Drupal\track_usage\Trait\EntityUtilityTrait;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Default implementation for `track_usage.tracker` service.
 */
class Tracker implements TrackerInterface {

  use EntityUtilityTrait;

  public function __construct(
    protected readonly TrackPluginManagerInterface $trackManager,
    protected readonly ConfigFactoryInterface $configFactory,
    #[Autowire(service: 'track_usage.logger')]
    protected readonly LoggerChannelInterface $logger,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function track(EntityInterface $entity, TrackConfigInterface $config): UsageCollection {
    $usages = new UsageCollection();
    if ($config->isSource($entity)) {
      assert($entity instanceof FieldableEntityInterface);
      $this->doTrack($usages, $entity, $config, [$this->getKey($entity)]);
    }
    return $usages;
  }

  /**
   * Tracks recursively the path to target entities.
   *
   * @param \Drupal\track_usage\Model\UsageCollection $usages
   *   The usage collection.
   * @param \Drupal\Core\Entity\FieldableEntityInterface $entity
   *   The source or traversable entity.
   * @param \Drupal\track_usage\Entity\TrackConfigInterface $config
   *   The track configuration.
   * @param string[] $path
   *   The path to the current processed entity.
   * @param array<array-key, string[]> $visited
   *   The visited entities. Used to avoid infinite recursion.
   */
  protected function doTrack(
    UsageCollection $usages,
    FieldableEntityInterface $entity,
    TrackConfigInterface $config,
    array $path = [],
    array $visited = [],
  ): void {
    // Limit deep traversing.
    if (count($path) > 20) {
      $this->logger->warning("The @type entity with ID '@id' has an unusual depth traversing on path: @path", [
        '@type' => $entity->getEntityType()->getSingularLabel(),
        '@id' => $entity->id(),
        '@path' => "'" . implode("', '", $path) . "'",
      ]);
      return;
    }

    $key = $this->getKey($entity);
    if (!isset($visited[$key])) {
      $visited[$key] = $path;
    }
    else {
      // Recursion detected.
      // @todo Needs testing.
      return;
    }

    foreach ($this->getFields($entity) as $itemList) {
      $fieldType = $itemList->getFieldDefinition()->getType();
      $trackPlugins = $this->trackManager->getApplicablePlugins($fieldType);
      foreach ($trackPlugins as $trackPlugin) {
        foreach ($trackPlugin->getTargetEntities($itemList) as $targetEntity) {
          if ($config->isTarget($targetEntity)) {
            $usages->add($targetEntity, $path);
          }

          if ($config->isTraversable($targetEntity)) {
            $this->doTrack(
              $usages,
              $targetEntity,
              $config,
              [...$path, $this->getKey($targetEntity)],
              $visited,
            );
          }
        }
      }
    }
  }

  /**
   * Returns non-empty entity fields.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity.
   *
   * @return iterable<\Drupal\Core\Field\FieldItemListInterface>
   *   A list of non-empty entity fields.
   */
  protected function getFields(FieldableEntityInterface $entity): iterable {
    foreach ($entity->getFields() as $fieldItemList) {
      if ($this->shouldFollowField($fieldItemList)) {
        yield $fieldItemList;
      }
    }
  }

  /**
   * Checks if a field should be followed.
   *
   * @param \Drupal\Core\Field\FieldItemListInterface $fieldItemList
   *   The field item list.
   *
   * @return bool
   *   Whether the field should be followed.
   */
  protected function shouldFollowField(FieldItemListInterface $fieldItemList): bool {
    return !$fieldItemList->isEmpty() &&
      in_array(
        $fieldItemList->getFieldDefinition()->getType(),
        $this->trackManager->getApplicableFieldTypes(),
        TRUE,
      );
  }

}
