<?php

declare(strict_types=1);

namespace Drupal\Tests\track_usage\Kernel\Track;

use Drupal\block_content\Entity\BlockContent;
use Drupal\block_content\Entity\BlockContentType;
use Drupal\entity_browser\Entity\EntityBrowser;
use Drupal\entity_test\Entity\EntityTest;
use Drupal\layout_builder\Entity\LayoutBuilderEntityViewDisplay;
use Drupal\Tests\block_field\Traits\BlockFieldTestTrait;

/**
 * Tests the 'block_field' track plugin.
 *
 * @group track_usage
 */
class BlockFieldTest extends TrackTestBase {

  use BlockFieldTestTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'block_content',
    'block_field',
    'entity_browser',
    'entity_browser_block',
    'entity_browser_entity_form',
    'layout_builder',
    'layout_discovery',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('block_content');
    $this->installEntitySchema('entity_test');
    $this->installEntitySchema('entity_browser');
    BlockContentType::create(['id' => 'basic', 'label' => 'Basic'])->save();
    $this->createBlockField('entity_test', 'entity_test', 'block', 'Block');

    LayoutBuilderEntityViewDisplay::create([
      'targetEntityType' => 'entity_test',
      'bundle' => 'entity_test',
      'mode' => 'default',
      'status' => TRUE,
    ])->enableLayoutBuilder()->setOverridable()->save();

    EntityBrowser::create([
      'name' => 'test',
      'display' => 'modal',
      'selection_display' => 'no_display',
      'widget_selector' => 'tabs',
      'widgets' => [
        $this->container->get('uuid')->generate() => [
          'id' => 'entity_form',
          'settings' => [
            'entity_type' => 'entity_test',
            'bundle' => 'entity_test',
          ],
        ],
      ],
    ])->save();
  }

  /**
   * @covers \Drupal\track_usage\Plugin\TrackUsage\Track\BlockField
   */
  public function testPlugin(): void {
    $block1 = BlockContent::create(['info' => 'Foo', 'type' => 'basic']);
    $block1->save();
    $block2 = BlockContent::create(['info' => 'Baz', 'type' => 'basic']);
    $block2->save();

    $entity1 = EntityTest::create([
      'type' => 'entity_test',
      'name' => $this->randomString(),
    ]);
    $entity1->save();
    $entity2 = EntityTest::create([
      'type' => 'entity_test',
      'name' => $this->randomString(),
    ]);
    $entity2->save();

    $entity = EntityTest::create([
      'type' => 'entity_test',
      'name' => $this->randomString(),
      'block' => [
        ['plugin_id' => 'block_content:' . $block1->uuid()],
        ['plugin_id' => 'block_content:' . $block2->uuid()],
        // Duplicates are not tracked.
        ['plugin_id' => 'block_content:' . $block1->uuid()],
        // Blocks other than block_content are ignored.
        ['plugin_id' => 'system_branding_block'],
        // Entity Browser Block.
        [
          'plugin_id' => 'entity_browser_block:test',
          'settings' => [
            'entity_ids' => [
              "entity_test:{$entity1->id()}",
              "entity_test:{$entity2->id()}",
            ],
          ],
        ],
      ],
    ]);
    $entity->save();

    $this->assertTargetEntities('block_field', $entity, 'block', [
      $block1,
      $block2,
      $entity1,
      $entity2,
    ]);
  }

}
