<?php

declare(strict_types=1);

namespace Drupal\Tests\track_usage\Kernel\Track;

use Drupal\entity_test\Entity\EntityTest;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\file\Entity\File;
use Drupal\file\FileInterface;
use Drupal\filter\Entity\FilterFormat;
use Drupal\Tests\TestFileCreationTrait;

/**
 * Tests the 'ckeditor_image' track plugin.
 *
 * @group track_usage
 */
class CkEditorImageTest extends TrackTestBase {

  use TestFileCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'filter',
    'text',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('entity_test');
    $this->installEntitySchema('file');
    $this->installSchema('file', ['file_usage']);

    FieldStorageConfig::create([
      'field_name' => 'text',
      'type' => 'text_with_summary',
      'entity_type' => 'entity_test',
      'cardinality' => 1,
    ])->save();
    FieldConfig::create([
      'field_name' => 'text',
      'label' => $this->randomString(),
      'entity_type' => 'entity_test',
      'bundle' => 'entity_test',
    ])->save();
    FilterFormat::create([
      'format' => 'basic_html',
      'name' => 'Basic HTML',
    ])->setFilterConfig('filter_html', [
      'status' => TRUE,
      'settings' => [
        'allowed_html' => '<img data-entity-uuid data-entity-type>',
      ],
    ])->save();
  }

  /**
   * @covers \Drupal\track_usage\Plugin\TrackUsage\Track\CkEditorImage
   */
  public function testPlugin(): void {
    $files = $this->getTestFiles('image');

    $image1 = File::create(['uri' => $files[0]->uri]);
    $image1->save();
    $image2 = File::create(['uri' => $files[1]->uri]);
    $image2->save();
    $image3 = File::create(['uri' => $files[2]->uri]);
    $image3->save();
    $image4 = File::create(['uri' => $files[3]->uri]);
    $image4->save();
    $image5 = File::create(['uri' => $files[4]->uri]);
    $image5->save();

    $text = $this->randomString() .
      $this->buildImageTag($image1) . $this->randomString() .
      $this->buildImageTag($image2) . $this->randomString() .
      $this->buildImageTag($image3) . $this->randomString() .
      $this->buildImageTag($image4) . $this->randomString();
    $summary = $this->randomString() . $this->buildImageTag($image5) . $this->randomString();

    $entity = EntityTest::create([
      'type' => 'entity_test',
      'name' => $this->randomString(),
      'text' => [
        'value' => $text,
        'summary' => $summary,
        'format' => 'basic_html',
      ],
    ]);
    $entity->save();

    // Delete an image that is referred in text.
    $image3->delete();

    $this->assertTargetEntities('ckeditor_image', $entity, 'text', [
      $image1,
      $image2,
      $image4,
      $image5,
    ]);
  }

  /**
   * Builds an image HTML tag.
   *
   * @param \Drupal\file\FileInterface $file
   *   The file entity.
   *
   * @return string
   *   HTML tag.
   */
  protected function buildImageTag(FileInterface $file): string {
    $fileUrlGenerator = $this->container->get('file_url_generator');
    $src = $fileUrlGenerator->generate($file->getFileUri())->toString();
    return '<img src="' . $src . '" data-entity-type="file" data-entity-uuid="' . $file->uuid() . '" />';
  }

}
