<?php

declare(strict_types=1);

namespace Drupal\Tests\track_usage\Kernel\Track;

use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\entity_test\Entity\EntityTest;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\filter\Entity\FilterFormat;

/**
 * Tests the 'entity_embed' track plugin.
 *
 * @group track_usage
 */
class EntityEmbedTest extends TrackTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'filter',
    'text',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('entity_test');

    FieldStorageConfig::create([
      'field_name' => 'text',
      'type' => 'text_with_summary',
      'entity_type' => 'entity_test',
      'cardinality' => 1,
    ])->save();
    FieldConfig::create([
      'field_name' => 'text',
      'label' => $this->randomString(),
      'entity_type' => 'entity_test',
      'bundle' => 'entity_test',
    ])->save();
    FilterFormat::create([
      'format' => 'basic_html',
      'name' => 'Basic HTML',
    ])->setFilterConfig('filter_html', [
      'status' => TRUE,
      'settings' => [
        'allowed_html' => '<drupal-entity data-entity-uuid data-entity-type>',
      ],
    ])->save();
  }

  /**
   * @covers \Drupal\track_usage\Plugin\TrackUsage\Track\EntityEmbed
   */
  public function testPlugin(): void {
    $ref1 = EntityTest::create([
      'type' => 'entity_test',
      'name' => $this->randomString(),
    ]);
    $ref1->save();
    $ref2 = EntityTest::create([
      'type' => 'entity_test',
      'name' => $this->randomString(),
    ]);
    $ref2->save();
    $ref3 = EntityTest::create([
      'type' => 'entity_test',
      'name' => $this->randomString(),
    ]);
    $ref3->save();

    $text = $this->randomString() .
      $this->buildEntityEmbedTag($ref1) . $this->randomString() .
      $this->buildEntityEmbedTag($ref2) . $this->randomString();
    $summary = $this->randomString() .
      $this->buildEntityEmbedTag($ref3) . $this->randomString();

    $entity = EntityTest::create([
      'type' => 'entity_test',
      'name' => $this->randomString(),
      'text' => [
        'value' => $text,
        'summary' => $summary,
        'format' => 'basic_html',
      ],
    ]);
    $entity->save();

    $this->assertTargetEntities('entity_embed', $entity, 'text', [
      $ref1,
      $ref2,
      $ref3,
    ]);
  }

  /**
   * Builds a Drupal embed entity HTML tag.
   *
   * @param \Drupal\Core\Entity\FieldableEntityInterface $entity
   *   The entity.
   *
   * @return string
   *   HTML tag.
   */
  protected function buildEntityEmbedTag(FieldableEntityInterface $entity): string {
    return '<drupal-entity data-entity-type="' . $entity->getEntityTypeId() . '" data-entity-uuid="' . $entity->uuid() . '"></drupal-entity>';
  }

}
