<?php

declare(strict_types=1);

namespace Drupal\Tests\track_usage\Kernel\Track;

use Drupal\entity_test\Entity\EntityTest;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\file\Entity\File;
use Drupal\redirect\Entity\Redirect;
use Drupal\Tests\TestFileCreationTrait;

/**
 * Tests the 'link' track plugin.
 *
 * @group track_usage
 */
class LinkTest extends TrackTestBase {

  use TestFileCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('entity_test');
    $this->installEntitySchema('file');

    FieldStorageConfig::create([
      'type' => 'link',
      'entity_type' => 'entity_test',
      'field_name' => 'link',
    ])->save();
    FieldConfig::create([
      'entity_type' => 'entity_test',
      'bundle' => 'entity_test',
      'field_name' => 'link',
      'label' => $this->randomString(),
    ])->save();
  }

  /**
   * @covers \Drupal\track_usage\Plugin\TrackUsage\Track\Link
   */
  public function testPlugin(): void {
    $files = $this->getTestFiles('text');
    $file1 = File::create(['uri' => $files[0]->uri]);
    $file1->save();
    $file2 = File::create(['uri' => $files[1]->uri]);
    $file2->save();

    $ref1 = EntityTest::create([
      'type' => 'entity_test',
      'name' => $this->randomString(),
    ]);
    $ref1->save();
    $ref2 = EntityTest::create([
      'type' => 'entity_test',
      'name' => $this->randomString(),
    ]);
    $ref2->save();
    $ref3 = EntityTest::create([
      'type' => 'entity_test',
      'name' => $this->randomString(),
    ]);
    $ref3->save();

    $redirect = Redirect::create();
    $redirect->setSource('this/is/a/redirect');
    $redirect->setRedirect($ref3->toUrl()->toString());
    $redirect->save();

    $file2Path = $this->container->get('file_url_generator')->generateString($file2->getFileUri());

    $entity = EntityTest::create([
      'type' => 'entity_test',
      'name' => $this->randomString(),
      'link' => [
        ['uri' => 'internal:/' . $ref1->toUrl()->getInternalPath()],
        ['uri' => 'entity:entity_test/' . $ref2->id()],
        ['uri' => $GLOBALS['base_url'] . '/entity_test/' . $ref2->id()],
        ['uri' => $file1->getFileUri()],
        ['uri' => 'base:' . $file2Path],
        ['uri' => 'http://example.com'],
        ['uri' => $GLOBALS['base_url'] . '/does-not-exist'],
        ['uri' => 'base:/does-not-exist'],
        ['uri' => 'internal:/this/is/a/redirect'],
        ['uri' => 'mailto:example@example.com'],
      ],
    ]);
    $entity->save();

    $this->assertTargetEntities('link', $entity, 'link', [$ref1, $ref2, $ref3, $file1, $file2]);
  }

}
