<?php

declare(strict_types=1);

namespace Drupal\Tests\track_usage\Kernel\Track;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\KernelTests\KernelTestBase;
use Drupal\track_usage\Plugin\TrackUsage\TrackPluginInterface;
use Drupal\track_usage\Plugin\TrackUsage\TrackPluginManagerInterface;
use Drupal\track_usage\RecorderInterface;

/**
 * Base class for track plugin tests.
 */
abstract class TrackTestBase extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'entity_test',
    'field',
    'file',
    'link',
    'path_alias',
    'track_usage',
    'redirect',
    'system',
    'user',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('path_alias');
    $this->installEntitySchema('redirect');
    $this->installSchema('track_usage', [
      RecorderInterface::TABLE,
      RecorderInterface::TABLE_PATHS,
    ]);
    $this->setSetting('file_public_path', 'sites/default/files');
  }

  /**
   * Asserts that an entity targets a list of entities via a track plugin.
   *
   * @param string $pluginId
   *   The track plugin ID.
   * @param \Drupal\Core\Entity\FieldableEntityInterface $entity
   *   The entity.
   * @param string $fieldName
   *   The entity field.
   * @param list<\Drupal\Core\Entity\FieldableEntityInterface> $expectedTargetEntities
   *   A list of expected target entities.
   */
  protected function assertTargetEntities(string $pluginId, FieldableEntityInterface $entity, string $fieldName, array $expectedTargetEntities): void {
    $plugin = $this->container->get(TrackPluginManagerInterface::class)->createInstance($pluginId);
    assert($plugin instanceof TrackPluginInterface);
    $actualTargetEntities = iterator_to_array($plugin->getTargetEntities($entity->get($fieldName)));
    $this->assertCount(count($expectedTargetEntities), $actualTargetEntities);
    $this->assertSame($this->normalize($expectedTargetEntities), $this->normalize($actualTargetEntities));
  }

  /**
   * Normalizes a list of entities to facilitate easy comparison.
   *
   * @param list<\Drupal\Core\Entity\FieldableEntityInterface> $entities
   *   A list of entities.
   *
   * @return array<array-key, list<int|string>>
   *   A normalized list of entity IDs grouped by their entity type ID.
   */
  protected function normalize(array $entities): array {
    $entitiesByEntityType = [];
    foreach ($entities as $entity) {
      assert($entity instanceof EntityInterface);
      $entitiesByEntityType[$entity->getEntityTypeId()][] = (string) $entity->id();
    }
    array_walk($entitiesByEntityType, function (array $ids) {
      sort($ids);
    });
    ksort($entitiesByEntityType);

    return $entitiesByEntityType;
  }

}
