<?php

declare(strict_types=1);

namespace Drupal\track_usage\Entity;

use Drupal\Core\Config\Entity\ConfigEntityBase;
use Drupal\Core\Entity\Attribute\ConfigEntityType;
use Drupal\Core\Entity\EntityAccessControlHandler;
use Drupal\Core\Entity\EntityDeleteForm;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Entity\Routing\AdminHtmlRouteProvider;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\track_usage\Form\TrackConfigForm;
use Drupal\track_usage\Model\EntityRole;

/**
 * The track config entity class.
 *
 * @ConfigEntityType(
 *   id = "track_usage_config",
 *   label = @Translation("Track configuration'"),
 *   label_collection = @Translation("Track configurations"),
 *   label_singular = @Translation("track configuration"),
 *   label_plural = @Translation("track configurations"),
 *   label_plural = @Translation("track configurations"),
 *   config_prefix = "config",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *   },
 *   handlers = {
 *     "access" = "Drupal\Core\Entity\EntityAccessControlHandler",
 *     "form" = {
 *       "add" = "Drupal\track_usage\Form\TrackConfigForm",
 *       "edit" = "Drupal\track_usage\Form\TrackConfigForm",
 *       "delete" = "Drupal\Core\Entity\EntityDeleteForm",
 *     },
 *     "list_builder" = "Drupal\track_usage\Entity\TrackConfigListBuilder",
 *     "route_provider" = {
 *       "html" = "Drupal\Core\Entity\Routing\AdminHtmlRouteProvider",
 *     },
 *   },
 *   links = {
 *     "collection" = "/admin/config/system/usage-track",
 *     "edit-form" = "/admin/config/system/usage-track/manage/{track_usage_config}",
 *     "add-form" = "/admin/config/system/usage-track/add",
 *     "delete-form" = "/admin/config/system/usage-track/manage/{track_usage_config}/delete",
 *   },
 *   admin_permission = "administer track usage",
 *   label_count = {
 *     "singular" = "@count track configuration",
 *     "plural" = "@count track configurations",
 *   },
 *   config_export = {
 *     "id",
 *     "label",
 *     "trackPlugins",
 *     "activeRevision",
 *     "realTimeRecording",
 *     "source",
 *     "traversable",
 *     "target",
 *   },
 * )
 *
 * @phpstan-ignore-next-line
 */
#[ConfigEntityType(
  id: 'track_usage_config',
  label: new TranslatableMarkup('Track configuration'),
  label_collection: new TranslatableMarkup('Track configurations'),
  label_singular: new TranslatableMarkup('track configuration'),
  label_plural: new TranslatableMarkup('track configurations'),
  config_prefix: 'config',
  entity_keys: [
    'id' => 'id',
    'label' => 'label',
  ],
  handlers: [
    'access' => EntityAccessControlHandler::class,
    'form' => [
      'add' => TrackConfigForm::class,
      'edit' => TrackConfigForm::class,
      'delete' => EntityDeleteForm::class,
    ],
    'list_builder' => TrackConfigListBuilder::class,
    'route_provider' => [
      'html' => AdminHtmlRouteProvider::class,
    ],
  ],
  links: [
    'collection' => '/admin/config/system/usage-track',
    'edit-form' => '/admin/config/system/usage-track/manage/{track_usage_config}',
    'add-form' => '/admin/config/system/usage-track/add',
    'delete-form' => '/admin/config/system/usage-track/manage/{track_usage_config}/delete',
  ],
  admin_permission: 'administer track usage',
  label_count: [
    'singular' => '@count track configuration',
    'plural' => '@count track configurations',
  ],
  config_export: [
    'id',
    'label',
    'trackPlugins',
    'activeRevision',
    'realTimeRecording',
    'source',
    'traversable',
    'target',
  ],
)]
class TrackConfig extends ConfigEntityBase implements TrackConfigInterface {

  /**
   * Track config ID.
   */
  protected string $id;

  /**
   * Track config label.
   */
  protected string $label;

  /**
   * Enabled track plugins.
   *
   * @var list<non-empty-string>
   */
  protected array $trackPlugins = [];

  /**
   * Whether to track only the active revisions of source entity.
   */
  protected bool $activeRevision = TRUE;

  /**
   * Whether to record entity changes in real time.
   */
  protected bool $realTimeRecording = FALSE;

  /**
   * Keys are entity types, values are a list of bundles.
   *
   * @var array<array-key, list<non-empty-string>>
   */
  protected array $source = [];

  /**
   * Keys are entity types, values are a list of bundles.
   *
   * @var array<array-key, list<non-empty-string>>
   */
  protected array $traversable = [];

  /**
   * Keys are entity types, values are a list of bundles.
   *
   * @var array<array-key, list<non-empty-string>>
   */
  protected array $target = [];

  /**
   * {@inheritdoc}
   */
  public function getTrackPluginIds(): array {
    return $this->trackPlugins;
  }

  /**
   * {@inheritdoc}
   */
  public function useRealTimeRecording(): bool {
    return $this->realTimeRecording;
  }

  /**
   * {@inheritdoc}
   */
  public function trackOnlyActiveRevision(): bool {
    return $this->activeRevision;
  }

  /**
   * {@inheritdoc}
   */
  public function couldBeTargetableEntityType(string $entityTypeId): bool {
    return isset($this->traversable[$entityTypeId]) || isset($this->target[$entityTypeId]);
  }

  /**
   * {@inheritdoc}
   */
  public function getTargetableBundles(string $entityTypeId): ?array {
    $entityType = \Drupal::entityTypeManager()->getDefinition($entityTypeId);
    if ($entityType->hasKey('bundle')) {
      return [
        ...$this->traversable[$entityTypeId] ?? [],
        ...$this->target[$entityTypeId] ?? [],
      ] ?: NULL;
    }
    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function isSource(EntityInterface $entity): bool {
    if ($entity instanceof FieldableEntityInterface) {
      return $this->checkEntityRole($entity, EntityRole::Source);
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function isTraversable(EntityInterface $entity): bool {
    if ($entity instanceof FieldableEntityInterface) {
      return $this->checkEntityRole($entity, EntityRole::Traversable);
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function isTarget(EntityInterface $entity): bool {
    // Target could be any type of entity, even config.
    return $this->checkEntityRole($entity, EntityRole::Target);
  }

  /**
   * Provides a helper method to check the entity role.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   * @param \Drupal\track_usage\Model\EntityRole $role
   *   The entity role.
   *
   * @return bool
   *   Whether the entity is configured for a given role.
   */
  protected function checkEntityRole(EntityInterface $entity, EntityRole $role): bool {
    $list = $this->{$role->value};
    $entityTypeId = $entity->getEntityTypeId();
    $bundle = $entity->bundle();
    return isset($list[$entityTypeId]) && (empty($list[$entityTypeId]) || in_array($bundle, $list[$entityTypeId], TRUE));
  }

}
