<?php

declare(strict_types=1);

namespace Drupal\track_usage\Hook;

use Drupal\comment\CommentInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\Entity\ConfigEntityStorageInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\track_usage\Entity\TrackConfigInterface;
use Drupal\track_usage\Model\Operation;
use Drupal\track_usage\RecorderInterface;
use Drupal\track_usage\Trait\BackwardsCompatibilityTrait;
use Drupal\track_usage\Trait\EntityUtilityTrait;

/**
 * Records usage on insert and update.
 */
class RecordUsageHook {

  use BackwardsCompatibilityTrait;
  use EntityUtilityTrait;

  public function __construct(
    protected readonly ConfigFactoryInterface $configFactory,
    protected readonly RecorderInterface $recorder,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly Connection $db,
  ) {}

  /**
   * Records usages for entity.
   */
  #[Hook('entity_insert')]
  #[Hook('entity_update')]
  public function record(EntityInterface $entity): void {
    foreach ($this->getRealTimeConfigs() as $config) {
      assert($config instanceof TrackConfigInterface);
      if (!$config->isSource($entity) && !$config->isTraversable($entity) && !$config->isTarget($entity)) {
        // The entity means nothing for this configuration context.
        continue;
      }

      if ($config->trackOnlyActiveRevision() && $config->isSource($entity) && $this->isRevisionable($entity)) {
        if (!$entity->isDefaultRevision()) {
          // Don't record if we're only looking for the default revisions.
          continue;
        }
        $this->recorder->cleanup($entity, Operation::Delete);
      }

      $this->recorder->record($entity, $config);
    }
  }

  /**
   * Removes usages for a deleted entity.
   */
  #[Hook('entity_delete')]
  public function onEntityDelete(EntityInterface $entity): void {
    $this->recorder->cleanup($entity, Operation::Delete);
  }

  /**
   * Removes usages for a deleted entity translation or language.
   */
  #[Hook('entity_translation_delete')]
  public function onEntityTranslationDelete(EntityInterface $entity): void {
    $this->recorder->cleanup($entity, Operation::DeleteTranslation);
  }

  /**
   * Removes usages for a deleted entity revision.
   */
  #[Hook('entity_revision_delete')]
  public function onEntityRevisionDelete(EntityInterface $entity): void {
    $this->recorder->cleanup($entity, Operation::DeleteRevision);
  }

  /**
   * Reacts on comment insertion.
   *
   * Comments are exposing a reference to the commented entity (usually node).
   * This is the reverse of the common case, when the reference is from top to
   * downstream entities (e.g., entity reference, etc.). On a new comment, the
   * commented entity is not aware, thus is not updated. This means, it doesn't
   * recompute its usages. We queue again the commented entity so it gets the
   * potential new usages.
   */
  #[Hook('comment_insert')]
  public function onCommentInsert(CommentInterface $comment): void {
    foreach ($this->getRealTimeConfigs() as $config) {
      if ($config->isTraversable($comment) || $config->isTarget($comment)) {
        $entity = $comment->getCommentedEntity();
        if ($entity && $config->isSource($entity)) {
          $this->recorder->record($entity, $config);
        }
      }
    }
  }

  /**
   * Returns a list of real-time Usage Track configurations.
   *
   * @return array<array-key, \Drupal\track_usage\Entity\TrackConfigInterface>
   *   List of real-time Usage Track configurations, keyed by configuration ID.
   */
  protected function getRealTimeConfigs(): array {
    $storage = $this->entityTypeManager->getStorage('track_usage_config');
    assert($storage instanceof ConfigEntityStorageInterface);

    $ids = $storage->getQuery()
      ->condition('status', TRUE)
      ->condition('realTimeRecording', TRUE)
      ->execute();

    return $ids ? $storage->loadMultiple($ids) : [];
  }

}
